@extends('layouts.app')

@section('title', $property->title)
@section('description', substr($property->description, 0, 160))

@section('content')
<div class="max-w-7xl mx-auto px-4 py-6">
    <!-- Back Button -->
    <div class="mb-4">
        <a href="{{ route('properties.index') }}" class="text-primary hover:text-primary-dark font-semibold">
            <i class="fas fa-arrow-left mr-2"></i> Back to Properties
        </a>
    </div>

    <!-- Hero Section with Photos -->
    <div class="mb-6">
        @if($property->photos && count($property->photos) > 0)
            <div class="grid grid-cols-4 gap-2 rounded-xl overflow-hidden">
                <!-- Main large photo -->
                <div class="col-span-4 md:col-span-2 md:row-span-2">
                    <img src="{{ asset('storage/' . $property->photos[0]) }}" 
                        alt="{{ $property->title }}"
                        class="w-full h-full object-cover cursor-pointer hover:opacity-90 transition"
                        onclick="openGallery(0)">
                </div>
                <!-- Smaller photos -->
                @foreach(array_slice($property->photos, 1, 4) as $index => $photo)
                    <div class="col-span-2 md:col-span-1">
                        <img src="{{ asset('storage/' . $photo) }}" 
                            alt="{{ $property->title }}"
                            class="w-full h-48 object-cover cursor-pointer hover:opacity-90 transition"
                            onclick="openGallery({{ $index + 1 }})">
                    </div>
                @endforeach
                
                @if(count($property->photos) > 5)
                    <button onclick="openGallery(0)" 
                        class="absolute bottom-4 right-4 bg-white px-4 py-2 rounded-lg shadow-lg hover:bg-gray-100 font-semibold">
                        <i class="fas fa-images mr-2"></i> Show all {{ count($property->photos) }} photos
                    </button>
                @endif
            </div>
        @endif
    </div>

    <div class="grid lg:grid-cols-3 gap-8">
        <!-- Main Content -->
        <div class="lg:col-span-2 space-y-6">
            <!-- Title & Location -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        @if($property->isFeatured())
                            <span class="inline-block bg-gradient-to-r from-yellow-400 to-orange-500 text-white px-3 py-1 rounded-full text-xs font-bold mb-2">
                                <i class="fas fa-star mr-1"></i> FEATURED
                            </span>
                        @endif
                        <h1 class="text-3xl font-bold text-secondary mb-2">{{ $property->title }}</h1>
                        <p class="text-gray-600 flex items-center">
                            <i class="fas fa-map-marker-alt text-primary mr-2"></i>
                            @auth
                                @php
                                    $userBooking = $property->bookings()
                                        ->where('user_id', auth()->id())
                                        ->where('payment_status', 'paid')
                                        ->where('admin_verified', true)
                                        ->first();
                                @endphp
                                @if($userBooking && $userBooking->map_visible)
                                    {{ $property->address }}, {{ $property->city }}, {{ $property->state }}
                                @else
                                    {{ $property->city }}, {{ $property->state }}
                                    <span class="text-sm text-gray-500 ml-2">(Full address after booking)</span>
                                @endif
                            @else
                                {{ $property->city }}, {{ $property->state }}
                                <span class="text-sm text-gray-500 ml-2">(Full address after booking)</span>
                            @endauth
                        </p>
                    </div>
                    
                    <!-- Favorite Button -->
                    @auth
                        <button onclick="toggleFavorite({{ $property->id }}, this)" 
                            class="ml-4 w-12 h-12 bg-gray-100 rounded-full flex items-center justify-center hover:scale-110 transition">
                            <i class="fas fa-heart text-xl {{ auth()->user()->hasFavorited($property->id) ? 'text-red-500' : 'text-gray-400' }}"></i>
                        </button>
                    @else
                        <a href="{{ route('login') }}" 
                            class="ml-4 w-12 h-12 bg-gray-100 rounded-full flex items-center justify-center hover:scale-110 transition">
                            <i class="fas fa-heart text-xl text-gray-400"></i>
                        </a>
                    @endauth
                </div>

                <!-- Badges -->
                <div class="flex flex-wrap gap-2">
                    @if($property->quality_standard)
                        <span class="bg-blue-100 text-blue-800 px-3 py-1 text-sm rounded-full font-semibold">
                            <i class="fas fa-certificate mr-1"></i> Quality Standard
                        </span>
                    @endif
                    @if($property->verified)
                        <span class="bg-green-100 text-green-800 px-3 py-1 text-sm rounded-full font-semibold">
                            <i class="fas fa-check-circle mr-1"></i> Verified
                        </span>
                    @endif
                    @if($property->quality_assured)
                        <span class="bg-purple-100 text-purple-800 px-3 py-1 text-sm rounded-full font-semibold">
                            <i class="fas fa-shield-alt mr-1"></i> Quality Assured
                        </span>
                    @endif
                    @if($property->allow_parties)
                        <span class="bg-primary/10 text-primary px-3 py-1 text-sm rounded-full font-semibold">
                            <i class="fas fa-glass-cheers mr-1"></i> Party Friendly
                        </span>
                    @endif
                </div>
            </div>

            <!-- Property Details Grid -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-xl font-bold text-secondary mb-4 flex items-center">
                    <i class="fas fa-info-circle text-primary mr-3"></i> Property Details
                </h2>
                <div class="grid grid-cols-2 md:grid-cols-3 gap-4">
                    <div class="bg-primary/5 rounded-lg p-4 text-center">
                        <i class="fas fa-home text-primary text-2xl mb-2"></i>
                        <p class="text-xs text-gray-600">Type</p>
                        <p class="font-bold text-secondary">{{ $property->house_type }}</p>
                    </div>
                    <div class="bg-primary/5 rounded-lg p-4 text-center">
                        <i class="fas fa-bed text-primary text-2xl mb-2"></i>
                        <p class="text-xs text-gray-600">Bedrooms</p>
                        <p class="font-bold text-secondary">{{ $property->bedrooms }}</p>
                    </div>
                    <div class="bg-primary/5 rounded-lg p-4 text-center">
                        <i class="fas fa-bed text-primary text-2xl mb-2"></i>
                        <p class="text-xs text-gray-600">Beds</p>
                        <p class="font-bold text-secondary">{{ $property->beds }}</p>
                    </div>
                    <div class="bg-primary/5 rounded-lg p-4 text-center">
                        <i class="fas fa-bath text-primary text-2xl mb-2"></i>
                        <p class="text-xs text-gray-600">Bathrooms</p>
                        <p class="font-bold text-secondary">{{ $property->bathrooms }}</p>
                    </div>
                    <div class="bg-primary/5 rounded-lg p-4 text-center">
                        <i class="fas fa-users text-primary text-2xl mb-2"></i>
                        <p class="text-xs text-gray-600">Max Guests</p>
                        <p class="font-bold text-secondary">{{ $property->max_guests }}</p>
                    </div>
                    <div class="bg-green-50 rounded-lg p-4 text-center">
                        <i class="fas fa-money-bill-wave text-green-600 text-2xl mb-2"></i>
                        <p class="text-xs text-gray-600">Per Night</p>
                        <p class="font-bold text-green-600">₦{{ number_format($property->price_per_night) }}</p>
                    </div>
                </div>
            </div>

            <!-- Description -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-xl font-bold text-secondary mb-4 flex items-center">
                    <i class="fas fa-align-left text-primary mr-3"></i> About This Place
                </h2>
                <p class="text-gray-700 whitespace-pre-line leading-relaxed">{{ $property->description }}</p>
                
                @if($property->highlighted_features)
                    <div class="mt-6">
                        <h3 class="font-bold text-lg mb-3">Highlighted Features</h3>
                        <div class="grid md:grid-cols-2 gap-3">
                            @foreach($property->highlighted_features as $feature)
                                <div class="flex items-start bg-green-50 rounded-lg p-3">
                                    <i class="fas fa-check-circle text-green-600 mt-1 mr-3"></i>
                                    <span class="text-gray-700">{{ $feature }}</span>
                                </div>
                            @endforeach
                        </div>
                    </div>
                @endif
            </div>

            <!-- Amenities -->
            @if($property->amenities)
                <div class="bg-white rounded-xl shadow-md p-6">
                    <h2 class="text-xl font-bold text-secondary mb-4 flex items-center">
                        <i class="fas fa-star text-primary mr-3"></i> Amenities
                    </h2>
                    <div class="grid md:grid-cols-2 gap-6">
                        @foreach($property->amenities as $category => $items)
                            <div>
                                <h3 class="font-semibold mb-3 flex items-center text-lg">
                                    @if($category === 'Essentials')
                                        <i class="fas fa-home text-primary mr-2"></i>
                                    @elseif($category === 'Kitchen & Dining')
                                        <i class="fas fa-utensils text-green-600 mr-2"></i>
                                    @elseif($category === 'Outdoor & Facilities')
                                        <i class="fas fa-tree text-primary mr-2"></i>
                                    @elseif($category === 'Safety & Other')
                                        <i class="fas fa-shield-alt text-red-600 mr-2"></i>
                                    @else
                                        <i class="fas fa-star text-yellow-600 mr-2"></i>
                                    @endif
                                    {{ $category }}
                                </h3>
                                <ul class="space-y-2">
                                    @foreach($items as $item)
                                        <li class="text-gray-700 flex items-center">
                                            <i class="fas fa-check text-green-600 mr-2"></i>
                                            {{ $item }}
                                        </li>
                                    @endforeach
                                </ul>
                            </div>
                        @endforeach
                    </div>
                </div>
            @endif

            <!-- House Rules -->
            @if($property->house_rules)
                <div class="bg-white rounded-xl shadow-md p-6">
                    <h2 class="text-xl font-bold text-secondary mb-4 flex items-center">
                        <i class="fas fa-list-check text-primary mr-3"></i> House Rules
                    </h2>
                    <ul class="space-y-3">
                        @foreach($property->house_rules as $rule)
                            <li class="flex items-start">
                                <i class="fas fa-circle text-primary text-xs mt-2 mr-3"></i>
                                <span class="text-gray-700">{{ $rule }}</span>
                            </li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <!-- Reviews -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-xl font-bold text-secondary mb-4 flex items-center">
                    <i class="fas fa-star text-primary mr-3"></i> Reviews
                    @if($averageRating)
                        <span class="ml-2 text-yellow-500">★ {{ number_format($averageRating, 1) }}</span>
                        <span class="ml-2 text-sm text-gray-600">({{ $property->reviews()->visible()->count() }} reviews)</span>
                    @endif
                </h2>

                @forelse($property->reviews()->visible()->latest()->take(5)->get() as $review)
                    <div class="border-b pb-4 mb-4 last:border-b-0">
                        <div class="flex items-start mb-2">
                            @if($review->user->profile_photo)
                                <img src="{{ asset('storage/' . $review->user->profile_photo) }}" 
                                    alt="{{ $review->user->name }}"
                                    class="w-12 h-12 rounded-full object-cover">
                            @else
                                <div class="w-12 h-12 bg-primary rounded-full flex items-center justify-center text-white font-bold text-lg">
                                    {{ substr($review->user->name, 0, 1) }}
                                </div>
                            @endif
                            <div class="ml-3 flex-1">
                                <div class="flex items-center justify-between">
                                    <div>
                                        <p class="font-semibold">{{ $review->user->name }}</p>
                                        <p class="text-sm text-gray-600">{{ $review->created_at->format('F Y') }}</p>
                                    </div>
                                    <div class="text-yellow-500 text-lg">
                                        {{ str_repeat('★', $review->rating) }}{{ str_repeat('☆', 5 - $review->rating) }}
                                    </div>
                                </div>
                                <p class="text-gray-700 mt-2">{{ $review->comment }}</p>
                            </div>
                        </div>
                    </div>
                @empty
                    <div class="text-center py-8">
                        <i class="fas fa-comments text-gray-300 text-4xl mb-3"></i>
                        <p class="text-gray-600">No reviews yet. Be the first to review!</p>
                    </div>
                @endforelse

                @if($property->reviews()->visible()->count() > 5)
                    <div class="text-center mt-4">
                        <button class="text-primary hover:text-primary-dark font-semibold">
                            View all {{ $property->reviews()->visible()->count() }} reviews
                        </button>
                    </div>
                @endif
            </div>
        </div>

        <!-- Booking Sidebar -->
        <div class="lg:col-span-1">
            <div class="bg-white rounded-xl shadow-lg p-6 sticky top-4">
                <div class="text-center mb-6">
                    <div class="text-4xl font-bold text-primary mb-2">
                        ₦{{ number_format($property->price_per_night) }}
                    </div>
                    <p class="text-gray-600">per night</p>
                </div>

                @auth
                    <a href="{{ route('bookings.create', $property) }}" 
                        class="block w-full bg-gradient-to-r from-primary to-primary-dark text-white text-center py-4 rounded-lg hover:shadow-xl font-bold text-lg transition transform hover:scale-105">
                        <i class="fas fa-calendar-check mr-2"></i> Book Now
                    </a>
                @else
                    <a href="{{ route('login') }}" 
                        class="block w-full bg-gradient-to-r from-primary to-primary-dark text-white text-center py-4 rounded-lg hover:shadow-xl font-bold text-lg transition transform hover:scale-105">
                        <i class="fas fa-sign-in-alt mr-2"></i> Login to Book
                    </a>
                @endauth

                <div class="mt-6 space-y-3">
                    <div class="flex justify-between items-center py-3 border-b">
                        <span class="text-gray-600">Caution Fee</span>
                        <span class="font-bold text-yellow-600">₦{{ number_format($property->caution_fee) }}</span>
                    </div>
                    <p class="text-xs text-gray-500">
                        <i class="fas fa-info-circle mr-1"></i>
                        Caution fee is fully refundable after stay
                    </p>
                </div>

                @if(!$property->allow_parties)
                    <div class="mt-4 bg-yellow-50 border-l-4 border-yellow-400 rounded-lg p-4">
                        <p class="text-sm text-yellow-800 font-semibold">
                            <i class="fas fa-ban mr-2"></i>
                            No parties or events allowed
                        </p>
                    </div>
                @endif

                <!-- Contact Host -->
                <div class="mt-6 pt-6 border-t">
                    <h3 class="font-bold text-secondary mb-3">Hosted by</h3>
                    <div class="flex items-center">
                        <div class="w-12 h-12 bg-primary rounded-full flex items-center justify-center text-white font-bold text-lg">
                            {{ substr($property->vendor->company_name, 0, 1) }}
                        </div>
                        <div class="ml-3">
                            <p class="font-semibold">{{ $property->vendor->company_name }}</p>
                            <p class="text-sm text-gray-600">Property Host</p>
                        </div>
                    </div>
                </div>

                <!-- Share -->
                <div class="mt-6 pt-6 border-t">
                    <button class="w-full border-2 border-gray-300 text-gray-700 py-3 rounded-lg hover:bg-gray-50 font-semibold transition">
                        <i class="fas fa-share-alt mr-2"></i> Share Property
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
function toggleFavorite(propertyId, button) {
    fetch(`/favorites/toggle/${propertyId}`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const icon = button.querySelector('i');
            if (data.favorited) {
                icon.classList.remove('text-gray-400');
                icon.classList.add('text-red-500');
            } else {
                icon.classList.remove('text-red-500');
                icon.classList.add('text-gray-400');
            }
        }
    })
    .catch(error => console.error('Error:', error));
}

function openGallery(index) {
    // Implement photo gallery modal here
    console.log('Open gallery at index:', index);
}
</script>
@endpush
@endsection
