@extends('layouts.app')

@section('title', 'Payment - ' . $booking->property->title)

@section('content')
<div class="max-w-4xl mx-auto">
    <h1 class="text-3xl font-bold mb-6">Complete Payment</h1>

    <div class="grid md:grid-cols-3 gap-8">
        <!-- Payment Method Selection -->
        <div class="md:col-span-2">
            <div class="bg-white rounded-lg shadow-md p-6 mb-6">
                <h2 class="text-xl font-bold mb-4">Select Payment Method</h2>

                @if(count($availablePaymentMethods) === 0)
                    <div class="bg-yellow-100 border border-yellow-400 text-yellow-800 px-4 py-3 rounded">
                        <i class="fas fa-exclamation-triangle mr-2"></i>
                        No payment methods are currently available. Please contact support.
                    </div>
                @else
                    <form method="POST" action="{{ route('bookings.process-payment', $booking) }}">
                        @csrf

                        <div class="space-y-4 mb-6">
                            @foreach($availablePaymentMethods as $method)
                                <label class="block">
                                    <input type="radio" name="payment_method" value="{{ $method['id'] }}" required
                                        class="mr-3" {{ $loop->first ? 'checked' : '' }}>
                                    <div class="inline-block border-2 rounded-lg p-4 cursor-pointer hover:border-primary hover:bg-primary/5 transition w-full"
                                        onclick="this.previousElementSibling.checked = true;">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <h3 class="font-bold text-lg">{{ $method['name'] }}</h3>
                                                <p class="text-sm text-gray-600">{{ $method['description'] }}</p>
                                            </div>
                                            @if($method['id'] === 'paystack')
                                                <div class="text-primary">
                                                    <i class="fas fa-credit-card text-3xl"></i>
                                                </div>
                                            @elseif($method['id'] === 'flutterwave')
                                                <div class="text-orange-600">
                                                    <i class="fas fa-wallet text-3xl"></i>
                                                </div>
                                            @elseif($method['id'] === 'bank_transfer')
                                                <div class="text-green-600">
                                                    <i class="fas fa-university text-3xl"></i>
                                                </div>
                                            @endif
                                        </div>
                                    </div>
                                </label>
                            @endforeach
                        </div>

                        @error('payment_method')
                            <p class="text-red-500 text-sm mb-4">{{ $message }}</p>
                        @enderror

                        @error('payment')
                            <div class="bg-red-100 border border-red-400 text-red-800 px-4 py-3 rounded mb-4">
                                {{ $message }}
                            </div>
                        @enderror

                        <!-- Discount Code -->
                        @if($booking->discount_code)
                            <!-- Discount Already Applied -->
                            <div class="mb-6">
                                <div class="bg-green-50 border-2 border-green-200 rounded-xl p-4">
                                    <div class="flex items-center justify-between mb-2">
                                        <label class="block text-sm font-semibold text-green-800">
                                            <i class="fas fa-check-circle text-green-600 mr-2"></i>Discount Code Applied
                                        </label>
                                        <span class="bg-green-600 text-white px-3 py-1 rounded-full text-xs font-bold">
                                            ACTIVE
                                        </span>
                                    </div>
                                    <div class="bg-white rounded-lg p-3 mb-3">
                                        <p class="text-xs text-gray-500 uppercase tracking-wide mb-1">Code</p>
                                        <code class="text-xl font-bold text-green-600 font-mono">{{ $booking->discount_code }}</code>
                                    </div>
                                    <div class="flex items-center justify-between text-sm">
                                        <span class="text-gray-700">You're saving:</span>
                                        <span class="text-green-700 font-bold text-lg">₦{{ number_format($booking->discount_amount) }}</span>
                                    </div>
                                </div>
                            </div>
                        @else
                            <!-- Apply Discount Code -->
                            <div class="mb-6">
                                <label class="block text-sm font-semibold text-gray-700 mb-2">
                                    <i class="fas fa-ticket-alt text-primary mr-2"></i>Discount Code (Optional)
                                </label>
                                <div class="flex gap-2">
                                    <input type="text" name="discount_code" id="discount_code" 
                                        class="flex-1 border-2 border-gray-300 rounded-lg px-4 py-3 focus:border-primary focus:outline-none transition uppercase"
                                        placeholder="Enter discount code" style="text-transform: uppercase;">
                                    <button type="button" onclick="validateDiscountCode()" 
                                        class="px-6 py-3 bg-primary text-white rounded-lg hover:bg-primary-dark transition font-semibold">
                                        Apply
                                    </button>
                                </div>
                                <div id="discount-message" class="mt-2 text-sm"></div>
                                <div id="discount-details" class="mt-2 hidden">
                                    <div class="bg-green-50 border-l-4 border-green-500 rounded-lg p-3">
                                        <p class="text-green-800 font-semibold">
                                            <i class="fas fa-check-circle mr-2"></i>
                                            <span id="discount-text"></span>
                                        </p>
                                    </div>
                                </div>
                                <p class="text-xs text-gray-500 mt-2">
                                    <i class="fas fa-info-circle mr-1"></i>
                                    Note: Discount codes can only be applied during booking creation
                                </p>
                            </div>
                        @endif

                        <button type="submit" 
                            class="w-full bg-primary text-white py-3 rounded-lg hover:bg-primary-dark font-semibold text-lg">
                            <i class="fas fa-lock mr-2"></i> Proceed to Payment
                        </button>
                    </form>
                @endif
            </div>

            <!-- Security Notice -->
            <div class="bg-light-bg rounded-lg p-4 text-sm text-gray-600">
                <i class="fas fa-shield-alt text-green-600 mr-2"></i>
                Your payment is secured with industry-standard encryption
            </div>
        </div>

        <!-- Booking Summary -->
        <div>
            <div class="bg-white rounded-lg shadow-md p-6 sticky top-4">
                <h3 class="font-bold text-lg mb-4">Booking Summary</h3>
                
                <img src="{{ isset($booking->property->photos[0]) ? asset('storage/' . $booking->property->photos[0]) : '/images/placeholder.jpg' }}" 
                    alt="{{ $booking->property->title }}" 
                    class="w-full h-32 object-cover rounded mb-4">
                
                <h4 class="font-semibold mb-2">{{ $booking->property->title }}</h4>
                <p class="text-sm text-gray-600 mb-4">{{ $booking->property->city }}, {{ $booking->property->state }}</p>

                <div class="border-t pt-4 space-y-2 text-sm">
                    <div class="flex justify-between">
                        <span>Check-in:</span>
                        <span class="font-semibold">{{ $booking->start_date->format('M d, Y') }}</span>
                    </div>
                    <div class="flex justify-between">
                        <span>Check-out:</span>
                        <span class="font-semibold">{{ $booking->end_date->format('M d, Y') }}</span>
                    </div>
                    <div class="flex justify-between">
                        <span>Guests:</span>
                        <span class="font-semibold">{{ $booking->guest_count }}</span>
                    </div>
                    <div class="flex justify-between">
                        <span>Nights:</span>
                        <span class="font-semibold">{{ $booking->nights }}</span>
                    </div>
                </div>

                <div class="border-t mt-4 pt-4">
                    <div class="flex justify-between mb-2">
                        <span>Subtotal:</span>
                        <span>₦{{ number_format($booking->subtotal) }}</span>
                    </div>
                    @if($booking->caution_fee_amount > 0)
                        <div class="flex justify-between mb-2">
                            <span>Caution Fee <span class="text-xs">(Refundable)</span>:</span>
                            <span>₦{{ number_format($booking->caution_fee_amount) }}</span>
                        </div>
                    @endif
                    <div id="discount-section" class="{{ $booking->discount_amount > 0 ? '' : 'hidden' }} mb-3 pb-3 border-b">
                        <div class="flex justify-between text-sm text-green-600">
                            <span>Discount:</span>
                            <span class="font-semibold">-₦<span id="discount-value">{{ number_format($booking->discount_amount ?? 0) }}</span></span>
                        </div>
                    </div>
                    <div class="flex justify-between font-bold text-lg border-t pt-2">
                        <span>Total:</span>
                        <span class="text-primary" id="total-amount">₦{{ number_format($booking->total_amount) }}</span>
                    </div>
                </div>

                <p class="text-xs text-gray-500 mt-4">
                    <i class="fas fa-info-circle mr-1"></i>
                    Reference: {{ $booking->payment_reference }}
                </p>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
const originalAmount = {{ $booking->subtotal + $booking->caution_fee_amount }};
let appliedDiscount = {{ $booking->discount_amount ?? 0 }};

function validateDiscountCode() {
    const code = document.getElementById('discount_code').value.trim();
    const messageEl = document.getElementById('discount-message');
    const detailsEl = document.getElementById('discount-details');
    
    if (!code) {
        messageEl.innerHTML = '<p class="text-red-600"><i class="fas fa-exclamation-circle mr-1"></i>Please enter a discount code</p>';
        detailsEl.classList.add('hidden');
        return;
    }
    
    messageEl.innerHTML = '<p class="text-gray-600"><i class="fas fa-spinner fa-spin mr-1"></i>Validating code...</p>';
    
    fetch('/api/v1/discount-codes/validate', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        credentials: 'same-origin',
        body: JSON.stringify({
            code: code,
            amount: originalAmount
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.valid) {
            messageEl.innerHTML = `<p class="text-green-600"><i class="fas fa-check-circle mr-1"></i>${data.message}</p>`;
            detailsEl.classList.remove('hidden');
            document.getElementById('discount-text').textContent = 
                `You save ₦${data.discount_amount.toLocaleString()}! New total: ₦${data.final_amount.toLocaleString()}`;
            
            // Update sidebar
            appliedDiscount = data.discount_amount;
            updateTotal();
        } else {
            messageEl.innerHTML = `<p class="text-red-600"><i class="fas fa-times-circle mr-1"></i>${data.message}</p>`;
            detailsEl.classList.add('hidden');
            appliedDiscount = 0;
            updateTotal();
        }
    })
    .catch(error => {
        console.error('Error:', error);
        messageEl.innerHTML = `<p class="text-red-600"><i class="fas fa-exclamation-circle mr-1"></i>Error: ${error.message}. Please try again.</p>`;
        detailsEl.classList.add('hidden');
        appliedDiscount = 0;
        updateTotal();
    });
}

function updateTotal() {
    const discountSection = document.getElementById('discount-section');
    const discountValue = document.getElementById('discount-value');
    const totalAmount = document.getElementById('total-amount');
    
    if (appliedDiscount > 0) {
        discountSection.classList.remove('hidden');
        discountValue.textContent = appliedDiscount.toLocaleString();
        const newTotal = originalAmount - appliedDiscount;
        totalAmount.textContent = '₦' + newTotal.toLocaleString();
    } else {
        discountSection.classList.add('hidden');
        totalAmount.textContent = '₦' + originalAmount.toLocaleString();
    }
}
</script>
@endpush

@endsection
