@extends('layouts.app')

@section('title', 'My Bookings')

@section('content')
<div class="max-w-6xl mx-auto px-3 sm:px-4">
    <div class="flex flex-col sm:flex-row items-start sm:items-center justify-between mb-4 sm:mb-6 gap-3">
        <h1 class="text-2xl sm:text-3xl font-bold">My Bookings</h1>
        <a href="{{ route('transactions.index') }}" class="w-full sm:w-auto bg-primary text-white px-4 sm:px-6 py-2 sm:py-3 rounded-lg hover:bg-primary-dark transition font-semibold text-center text-sm sm:text-base">
            <i class="fas fa-receipt mr-2"></i>View Transactions
        </a>
    </div>

    <!-- Tabs -->
    <div class="mb-4 sm:mb-6 border-b overflow-x-auto">
        <nav class="flex space-x-2 sm:space-x-4 min-w-max">
            <button onclick="showTab('properties')" id="tab-properties" class="tab-button px-3 sm:px-4 py-2 font-semibold border-b-2 border-primary text-primary text-sm sm:text-base whitespace-nowrap">
                <i class="fas fa-home mr-1 sm:mr-2"></i>Property Bookings ({{ $bookings->total() }})
            </button>
            <button onclick="showTab('cars')" id="tab-cars" class="tab-button px-3 sm:px-4 py-2 font-semibold border-b-2 border-transparent text-gray-600 hover:text-primary text-sm sm:text-base whitespace-nowrap">
                <i class="fas fa-car mr-1 sm:mr-2"></i>Car Bookings ({{ $carBookings->total() }})
            </button>
        </nav>
    </div>

    <!-- Property Bookings -->
    <div id="content-properties" class="tab-content">
        @forelse($bookings as $booking)
        <div class="bg-white rounded-lg shadow-md p-4 sm:p-6 mb-3 sm:mb-4">
            <div class="flex flex-col md:flex-row gap-4 sm:gap-6">
                <!-- Property Image -->
                <img src="{{ isset($booking->property->photos[0]) ? asset('storage/' . $booking->property->photos[0]) : '/images/placeholder.jpg' }}" 
                    alt="{{ $booking->property->title }}" 
                    class="w-full md:w-48 h-40 sm:h-48 object-cover rounded">

                <!-- Booking Details -->
                <div class="flex-1">
                    <div class="flex flex-col sm:flex-row justify-between items-start mb-2 gap-2">
                        <h2 class="font-bold text-lg sm:text-xl">{{ $booking->property->title }}</h2>
                        <span class="px-2 sm:px-3 py-1 rounded-full text-xs sm:text-sm font-semibold whitespace-nowrap
                            @if($booking->booking_status === 'confirmed') bg-green-100 text-green-800
                            @elseif($booking->booking_status === 'pending') bg-yellow-100 text-yellow-800
                            @elseif($booking->booking_status === 'cancelled') bg-red-100 text-red-800
                            @else bg-primary/10 text-blue-800
                            @endif">
                            {{ ucfirst($booking->booking_status) }}
                        </span>
                    </div>

                    <p class="text-gray-600 mb-2 sm:mb-3 text-sm sm:text-base">{{ $booking->property->city }}, {{ $booking->property->state }}</p>

                    <div class="grid grid-cols-2 gap-2 sm:gap-4 mb-3 sm:mb-4">
                        <div>
                            <p class="text-xs sm:text-sm text-gray-600">Check-in</p>
                            <p class="font-semibold text-sm sm:text-base">{{ $booking->start_date->format('M d, Y') }}</p>
                        </div>
                        <div>
                            <p class="text-xs sm:text-sm text-gray-600">Check-out</p>
                            <p class="font-semibold text-sm sm:text-base">{{ $booking->end_date->format('M d, Y') }}</p>
                        </div>
                        <div>
                            <p class="text-xs sm:text-sm text-gray-600">Guests</p>
                            <p class="font-semibold text-sm sm:text-base">{{ $booking->guest_count }}</p>
                        </div>
                        <div>
                            <p class="text-xs sm:text-sm text-gray-600">Total Amount</p>
                            <p class="font-semibold text-sm sm:text-base">₦{{ number_format($booking->total_amount) }}</p>
                        </div>
                    </div>

                    <div class="flex flex-col sm:flex-row gap-2 sm:gap-3">
                        <a href="{{ route('bookings.show', $booking) }}" 
                            class="bg-primary text-white px-3 sm:px-4 py-2 rounded hover:bg-primary-dark text-center text-sm sm:text-base">
                            View Details
                        </a>

                        @if($booking->payment_status === 'pending' && $booking->vendor_approval_status === 'approved')
                            <a href="{{ route('bookings.payment', $booking) }}" 
                                class="bg-green-600 text-white px-3 sm:px-4 py-2 rounded hover:bg-green-700 text-center text-sm sm:text-base">
                                Complete Payment
                            </a>
                        @endif

                        @if($booking->canReview())
                            <a href="{{ route('reviews.create', $booking) }}" 
                                class="bg-primary text-white px-3 sm:px-4 py-2 rounded hover:bg-primary-dark text-center text-sm sm:text-base">
                                Leave Review
                            </a>
                        @endif
                    </div>
                </div>
            </div>
        </div>
        @empty
            <div class="bg-white rounded-lg shadow-md p-8 sm:p-12 text-center">
                <i class="fas fa-calendar-times text-5xl sm:text-6xl text-gray-300 mb-3 sm:mb-4"></i>
                <h2 class="text-lg sm:text-xl font-bold mb-2">No Property Bookings Yet</h2>
                <p class="text-gray-600 mb-4 sm:mb-6 text-sm sm:text-base">Start exploring amazing properties!</p>
                <a href="{{ route('properties.index') }}" 
                    class="inline-block bg-primary text-white px-4 sm:px-6 py-2 sm:py-3 rounded hover:bg-primary-dark text-sm sm:text-base">
                    Browse Properties
                </a>
            </div>
        @endforelse

        <!-- Pagination -->
        <div class="mt-4 sm:mt-6">
            {{ $bookings->links() }}
        </div>
    </div>

    <!-- Car Bookings -->
    <div id="content-cars" class="tab-content hidden">
        @forelse($carBookings as $booking)
            <div class="bg-white rounded-lg shadow-md p-4 sm:p-6 mb-3 sm:mb-4">
                <div class="flex flex-col md:flex-row gap-4 sm:gap-6">
                    <!-- Car Image -->
                    <img src="{{ $booking->car->main_photo ? asset('storage/' . $booking->car->main_photo) : '/images/placeholder.jpg' }}" 
                        alt="{{ $booking->car->full_name }}" 
                        class="w-full md:w-48 h-40 sm:h-48 object-cover rounded">

                    <!-- Booking Details -->
                    <div class="flex-1">
                        <div class="flex flex-col sm:flex-row justify-between items-start mb-2 gap-2">
                            <div>
                                <h2 class="font-bold text-lg sm:text-xl">{{ $booking->car->full_name }}</h2>
                                <p class="text-xs sm:text-sm text-gray-600">{{ $booking->car->category->name }}</p>
                            </div>
                            <span class="px-2 sm:px-3 py-1 rounded-full text-xs sm:text-sm font-semibold whitespace-nowrap
                                @if($booking->status === 'confirmed') bg-green-100 text-green-800
                                @elseif($booking->status === 'pending') bg-yellow-100 text-yellow-800
                                @elseif($booking->status === 'active') bg-blue-100 text-blue-800
                                @elseif($booking->status === 'completed') bg-gray-100 text-gray-800
                                @elseif($booking->status === 'cancelled') bg-red-100 text-red-800
                                @else bg-primary/10 text-blue-800
                                @endif">
                                {{ ucfirst($booking->status) }}
                            </span>
                        </div>

                        <p class="text-gray-600 mb-2 sm:mb-3 text-sm sm:text-base">
                            <i class="fas fa-map-marker-alt text-green-600 mr-1"></i>
                            {{ $booking->car->city }}, {{ $booking->car->state }}
                        </p>

                        <div class="grid grid-cols-2 gap-2 sm:gap-4 mb-3 sm:mb-4">
                            <div>
                                <p class="text-xs sm:text-sm text-gray-600">Pickup Date</p>
                                <p class="font-semibold text-sm sm:text-base">{{ \Carbon\Carbon::parse($booking->pickup_date)->format('M d, Y') }}</p>
                            </div>
                            <div>
                                <p class="text-xs sm:text-sm text-gray-600">Return Date</p>
                                <p class="font-semibold text-sm sm:text-base">{{ \Carbon\Carbon::parse($booking->return_date)->format('M d, Y') }}</p>
                            </div>
                            <div>
                                <p class="text-xs sm:text-sm text-gray-600">Total Days</p>
                                <p class="font-semibold text-sm sm:text-base">{{ $booking->total_days }} {{ Str::plural('day', $booking->total_days) }}</p>
                            </div>
                            <div>
                                <p class="text-xs sm:text-sm text-gray-600">Total Amount</p>
                                <p class="font-semibold text-sm sm:text-base">₦{{ number_format($booking->total_amount) }}</p>
                            </div>
                        </div>

                        <div class="mb-3">
                            <span class="text-xs sm:text-sm px-2 py-1 rounded
                                @if($booking->payment_status === 'paid') bg-green-100 text-green-800
                                @elseif($booking->payment_status === 'pending_verification') bg-yellow-100 text-yellow-800
                                @else bg-gray-100 text-gray-800
                                @endif">
                                <i class="fas fa-credit-card mr-1"></i>
                                Payment: {{ ucfirst(str_replace('_', ' ', $booking->payment_status)) }}
                            </span>
                        </div>

                        <div class="flex flex-col sm:flex-row gap-2 sm:gap-3">
                            <a href="{{ route('cars.booking.show', $booking) }}" 
                                class="bg-green-600 text-white px-3 sm:px-4 py-2 rounded hover:bg-green-700 text-center text-sm sm:text-base">
                                <i class="fas fa-eye mr-1"></i> View Details
                            </a>

                            @if($booking->payment_status === 'pending')
                                <a href="{{ route('cars.booking.payment', $booking) }}" 
                                    class="bg-primary text-white px-3 sm:px-4 py-2 rounded hover:bg-primary-dark text-center text-sm sm:text-base">
                                    <i class="fas fa-credit-card mr-1"></i> Complete Payment
                                </a>
                            @endif

                            @if($booking->canBeReviewed())
                                <a href="{{ route('cars.reviews.create', $booking) }}" 
                                    class="bg-yellow-600 text-white px-3 sm:px-4 py-2 rounded hover:bg-yellow-700 text-center text-sm sm:text-base">
                                    <i class="fas fa-star mr-1"></i> Leave Review
                                </a>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        @empty
            <div class="bg-white rounded-lg shadow-md p-8 sm:p-12 text-center">
                <i class="fas fa-car text-5xl sm:text-6xl text-gray-300 mb-3 sm:mb-4"></i>
                <h2 class="text-lg sm:text-xl font-bold mb-2">No Car Bookings Yet</h2>
                <p class="text-gray-600 mb-4 sm:mb-6 text-sm sm:text-base">Start exploring our car rental options!</p>
                <a href="{{ route('cars.index') }}" 
                    class="inline-block bg-green-600 text-white px-4 sm:px-6 py-2 sm:py-3 rounded hover:bg-green-700 text-sm sm:text-base">
                    <i class="fas fa-car mr-2"></i> Browse Cars
                </a>
            </div>
        @endforelse

        <!-- Pagination -->
        <div class="mt-4 sm:mt-6">
            {{ $carBookings->links() }}
        </div>
    </div>
</div>

<script>
function showTab(tab) {
    // Hide all tab contents
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.add('hidden');
    });
    
    // Remove active state from all tabs
    document.querySelectorAll('.tab-button').forEach(button => {
        button.classList.remove('border-primary', 'text-primary');
        button.classList.add('border-transparent', 'text-gray-600');
    });
    
    // Show selected tab content
    document.getElementById('content-' + tab).classList.remove('hidden');
    
    // Add active state to selected tab
    const activeTab = document.getElementById('tab-' + tab);
    activeTab.classList.remove('border-transparent', 'text-gray-600');
    activeTab.classList.add('border-primary', 'text-primary');
}
</script>
@endsection