@extends('layouts.app')

@section('title', 'Book ' . $property->title)

@section('content')
<div class="max-w-6xl mx-auto px-3 sm:px-4 py-4 sm:py-8">
    <!-- Back Button -->
    <div class="mb-3 sm:mb-4">
        <a href="{{ route('properties.show', $property) }}" class="text-primary hover:text-primary-dark font-semibold text-sm sm:text-base">
            <i class="fas fa-arrow-left mr-1 sm:mr-2"></i> Back to Property
        </a>
    </div>

    <!-- Header -->
    <div class="bg-gradient-to-r from-primary to-primary-dark text-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6 mb-4 sm:mb-6">
        <h1 class="text-xl sm:text-2xl md:text-3xl font-bold mb-2">
            <i class="fas fa-calendar-check mr-2 sm:mr-3"></i>Complete Your Booking
        </h1>
        <p class="text-primary-light text-sm sm:text-base">Just a few more steps to secure your stay</p>
    </div>

    <div class="grid lg:grid-cols-3 gap-4 sm:gap-6 lg:gap-8">
        <!-- Booking Form -->
        <div class="lg:col-span-2">
            <div class="bg-white rounded-lg sm:rounded-xl shadow-md p-4 sm:p-6">
                <h2 class="text-lg sm:text-xl font-bold text-secondary mb-4 sm:mb-6 flex items-center">
                    <i class="fas fa-edit text-primary mr-2 sm:mr-3 text-sm sm:text-base"></i> 
                    <span class="text-base sm:text-xl">Booking Details</span>
                </h2>

            <form method="POST" action="{{ route('bookings.store', $property) }}">
                @csrf

                <div class="mb-4 sm:mb-6">
                    <label class="block text-xs sm:text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-calendar-plus text-primary mr-1 sm:mr-2 text-xs sm:text-sm"></i>Check-in Date
                    </label>
                    <input type="date" name="start_date" id="check-in-date" required min="{{ date('Y-m-d', strtotime('+1 day')) }}"
                        class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2.5 sm:py-3 text-sm sm:text-base focus:border-primary focus:outline-none transition @error('start_date') border-red-500 @enderror"
                        value="{{ old('start_date') }}">
                    @error('start_date')
                        <p class="text-red-500 text-xs sm:text-sm mt-1"><i class="fas fa-exclamation-circle mr-1"></i>{{ $message }}</p>
                    @enderror
                    <p class="text-xs sm:text-sm text-gray-600 mt-1" id="checkin-warning"></p>
                </div>

                <div class="mb-4 sm:mb-6">
                    <label class="block text-xs sm:text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-calendar-minus text-primary mr-1 sm:mr-2 text-xs sm:text-sm"></i>Check-out Date
                    </label>
                    <input type="date" name="end_date" id="check-out-date" required
                        class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2.5 sm:py-3 text-sm sm:text-base focus:border-primary focus:outline-none transition @error('end_date') border-red-500 @enderror"
                        value="{{ old('end_date') }}">
                    @error('end_date')
                        <p class="text-red-500 text-xs sm:text-sm mt-1"><i class="fas fa-exclamation-circle mr-1"></i>{{ $message }}</p>
                    @enderror
                    <p class="text-xs sm:text-sm text-gray-600 mt-1" id="checkout-warning"></p>
                </div>

                <div class="bg-blue-50 border-l-4 border-blue-500 rounded-lg p-3 sm:p-4 mb-4 sm:mb-6">
                    <p class="text-xs sm:text-sm text-blue-800 font-semibold">
                        <i class="fas fa-info-circle mr-2"></i>
                        <span id="availability-status">Loading availability...</span>
                    </p>
                </div>

                <div class="grid sm:grid-cols-2 gap-3 sm:gap-6 mb-4 sm:mb-6">
                    <div>
                        <label class="block text-xs sm:text-sm font-semibold text-gray-700 mb-2">
                            <i class="fas fa-user text-primary mr-1 sm:mr-2 text-xs sm:text-sm"></i>Number of Adults
                        </label>
                        <input type="number" name="adults_count" min="1" required
                            class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2.5 sm:py-3 text-sm sm:text-base focus:border-primary focus:outline-none transition"
                            value="{{ old('adults_count', 1) }}">
                    </div>

                    <div>
                        <label class="block text-xs sm:text-sm font-semibold text-gray-700 mb-2">
                            <i class="fas fa-child text-primary mr-1 sm:mr-2 text-xs sm:text-sm"></i>Number of Children
                        </label>
                        <input type="number" name="children_count" min="0" required
                            class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2.5 sm:py-3 text-sm sm:text-base focus:border-primary focus:outline-none transition"
                            value="{{ old('children_count', 0) }}">
                    </div>
                </div>

                <div class="mb-4 sm:mb-6">
                    <label class="block text-xs sm:text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-suitcase text-primary mr-1 sm:mr-2 text-xs sm:text-sm"></i>Trip Type
                    </label>
                    <select name="trip_type" required class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2.5 sm:py-3 text-sm sm:text-base focus:border-primary focus:outline-none transition">
                        <option value="normal" {{ old('trip_type') == 'normal' ? 'selected' : '' }}>Normal Stay</option>
                        @if($property->allow_parties)
                            <option value="party" {{ old('trip_type') == 'party' ? 'selected' : '' }}>Party/Event</option>
                        @endif
                    </select>
                </div>

                <div class="bg-yellow-50 border-l-4 border-yellow-400 rounded-lg p-3 sm:p-4 mb-4 sm:mb-6">
                    <p class="text-xs sm:text-sm text-yellow-800 font-semibold">
                        <i class="fas fa-info-circle mr-2"></i>
                        If you're booking for a party or exceeding max guests ({{ $property->max_allowed_guests }}), 
                        vendor approval will be required before payment.
                    </p>
                </div>

                <!-- Discount Code -->
                <div class="mb-4 sm:mb-6">
                    <label class="block text-xs sm:text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-tag text-primary mr-1 sm:mr-2 text-xs sm:text-sm"></i>Discount Code (Optional)
                    </label>
                    <div class="flex flex-col sm:flex-row gap-2">
                        <input type="text" name="discount_code" id="discount_code" 
                            class="flex-1 border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2.5 sm:py-3 text-sm sm:text-base focus:border-primary focus:outline-none transition uppercase"
                            placeholder="Enter discount code" style="text-transform: uppercase;">
                        <button type="button" onclick="validateDiscountCode()" 
                            class="w-full sm:w-auto px-4 sm:px-6 py-2.5 sm:py-3 bg-primary text-white rounded-lg hover:bg-primary-dark transition font-semibold text-sm sm:text-base whitespace-nowrap">
                            Apply
                        </button>
                    </div>
                    <div id="discount-message" class="mt-2 text-xs sm:text-sm"></div>
                    <div id="discount-details" class="mt-2 hidden">
                        <div class="bg-green-50 border-l-4 border-green-500 rounded-lg p-2 sm:p-3">
                            <p class="text-green-800 font-semibold text-xs sm:text-sm">
                                <i class="fas fa-check-circle mr-2"></i>
                                <span id="discount-text"></span>
                            </p>
                        </div>
                    </div>
                </div>

                <button type="submit" class="w-full bg-gradient-to-r from-primary to-primary-dark text-white py-3 sm:py-4 rounded-lg hover:shadow-xl font-bold text-base sm:text-lg transition transform hover:scale-105">
                    <i class="fas fa-arrow-right mr-2"></i> Proceed to Book
                </button>
            </form>
            </div>
        </div>

        <!-- Property Summary Sidebar -->
        <div class="lg:col-span-1">
            <div class="bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6 lg:sticky lg:top-4">
                <h3 class="text-base sm:text-lg font-bold text-secondary mb-3 sm:mb-4 flex items-center">
                    <i class="fas fa-home text-primary mr-2 text-sm sm:text-base"></i> 
                    <span class="text-sm sm:text-lg">Property Summary</span>
                </h3>
                
                <img src="{{ isset($property->photos[0]) ? asset('storage/' . $property->photos[0]) : '/images/placeholder.jpg' }}" 
                    alt="{{ $property->title }}" 
                    class="w-full h-40 sm:h-48 object-cover rounded-lg mb-3 sm:mb-4">
                
                <h4 class="font-bold text-base sm:text-lg text-secondary mb-1">{{ $property->title }}</h4>
                <p class="text-gray-600 text-xs sm:text-sm mb-3 sm:mb-4 flex items-center">
                    <i class="fas fa-map-marker-alt text-primary mr-2"></i>
                    {{ $property->city }}, {{ $property->state }}
                </p>

                <!-- Property Details -->
                <div class="bg-gray-50 rounded-lg p-3 sm:p-4 mb-3 sm:mb-4">
                    <div class="grid grid-cols-2 gap-2 sm:gap-3 text-xs sm:text-sm">
                        <div class="flex items-center">
                            <i class="fas fa-bed text-primary mr-1.5 sm:mr-2 text-xs"></i>
                            <span>{{ $property->bedrooms }} Beds</span>
                        </div>
                        <div class="flex items-center">
                            <i class="fas fa-bath text-primary mr-1.5 sm:mr-2 text-xs"></i>
                            <span>{{ $property->bathrooms }} Baths</span>
                        </div>
                        <div class="flex items-center">
                            <i class="fas fa-users text-primary mr-1.5 sm:mr-2 text-xs"></i>
                            <span>{{ $property->max_guests }} Guests</span>
                        </div>
                        <div class="flex items-center">
                            <i class="fas fa-home text-primary mr-1.5 sm:mr-2 text-xs"></i>
                            <span class="truncate">{{ $property->house_type }}</span>
                        </div>
                    </div>
                </div>

                <!-- Pricing -->
                <div class="border-t pt-3 sm:pt-4 space-y-2 sm:space-y-3">
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600 text-xs sm:text-sm">Price per night</span>
                        <span class="font-bold text-primary text-base sm:text-lg">₦{{ number_format($property->price_per_night) }}</span>
                    </div>
                    <div id="nights-display" class="flex justify-between items-center hidden">
                        <span class="text-gray-600 text-xs sm:text-sm">
                            <span id="nights-count">0</span> night(s)
                        </span>
                        <span class="font-semibold text-sm sm:text-base" id="nights-total">₦0</span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600 text-xs sm:text-sm">Caution Fee <span class="text-xs">(Refundable)</span></span>
                        <span class="font-bold text-yellow-600 text-sm sm:text-base">₦{{ number_format($property->caution_fee) }}</span>
                    </div>
                </div>
                
                <div id="total-display" class="border-t mt-3 sm:mt-4 pt-3 sm:pt-4 hidden">
                    <div class="flex justify-between items-center">
                        <span class="text-base sm:text-lg font-bold text-gray-800">Total Amount</span>
                        <span class="text-xl sm:text-2xl font-bold text-primary" id="total-amount">₦0</span>
                    </div>
                    <p class="text-xs text-gray-500 mt-2">
                        <i class="fas fa-info-circle mr-1"></i>
                        Caution fee will be refunded after checkout
                    </p>
                </div>
                
                <div id="select-dates-notice" class="bg-blue-50 border-l-4 border-blue-500 rounded-lg p-2 sm:p-3 mt-3 sm:mt-4">
                    <p class="text-xs text-blue-800">
                        <i class="fas fa-info-circle mr-1"></i>
                        Select check-in and check-out dates to see the total amount.
                    </p>
                </div>

                <!-- Host Info -->
                <div class="mt-4 sm:mt-6 pt-4 sm:pt-6 border-t">
                    <h4 class="font-bold text-xs sm:text-sm text-gray-700 mb-3">Hosted by</h4>
                    <div class="flex items-center">
                        <div class="w-8 h-8 sm:w-10 sm:h-10 bg-primary rounded-full flex items-center justify-center text-white font-bold text-sm sm:text-base flex-shrink-0">
                            {{ substr($property->vendor->company_name, 0, 1) }}
                        </div>
                        <div class="ml-2 sm:ml-3 min-w-0">
                            <p class="font-semibold text-xs sm:text-sm truncate">{{ $property->vendor->company_name }}</p>
                            <p class="text-xs text-gray-600">Property Host</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let bookedDates = [];
let checkInInput = document.getElementById('check-in-date');
let checkOutInput = document.getElementById('check-out-date');
let submitButton = document.querySelector('button[type="submit"]');

// Fetch booked dates
async function fetchBookedDates() {
    try {
        const response = await fetch('/api/properties/{{ $property->id }}/booked-dates');
        const data = await response.json();
        bookedDates = data.booked_dates || [];
        console.log('Booked dates:', bookedDates);
        updateAvailabilityStatus();
    } catch (error) {
        console.error('Error fetching booked dates:', error);
        document.getElementById('availability-status').textContent = 'Unable to load availability. Please contact support.';
    }
}

function isDateBooked(dateStr) {
    return bookedDates.includes(dateStr);
}

function getDatesInRange(startDate, endDate) {
    const dates = [];
    const current = new Date(startDate);
    const end = new Date(endDate);
    
    while (current <= end) {
        dates.push(current.toISOString().split('T')[0]);
        current.setDate(current.getDate() + 1);
    }
    
    return dates;
}

function validateDates() {
    const checkIn = checkInInput.value;
    const checkOut = checkOutInput.value;
    
    const checkinWarning = document.getElementById('checkin-warning');
    const checkoutWarning = document.getElementById('checkout-warning');
    const statusElement = document.getElementById('availability-status');
    
    // Clear warnings
    checkinWarning.textContent = '';
    checkoutWarning.textContent = '';
    checkinWarning.className = 'text-xs sm:text-sm mt-1';
    checkoutWarning.className = 'text-xs sm:text-sm mt-1';
    
    if (!checkIn || !checkOut) {
        statusElement.textContent = 'Please select check-in and check-out dates.';
        statusElement.parentElement.className = 'text-xs sm:text-sm text-blue-800';
        return true;
    }
    
    // Check if check-out is after check-in
    if (new Date(checkOut) <= new Date(checkIn)) {
        checkoutWarning.textContent = '⚠️ Check-out must be after check-in date';
        checkoutWarning.className = 'text-xs sm:text-sm text-red-600 mt-1';
        statusElement.textContent = 'Invalid date range selected.';
        statusElement.parentElement.className = 'text-xs sm:text-sm text-red-800';
        submitButton.disabled = true;
        return false;
    }
    
    // Check if any dates in range are booked
    const datesInRange = getDatesInRange(checkIn, checkOut);
    const bookedInRange = datesInRange.filter(date => isDateBooked(date));
    
    if (bookedInRange.length > 0) {
        checkinWarning.textContent = '⚠️ Some dates in your selection are already booked';
        checkinWarning.className = 'text-xs sm:text-sm text-red-600 mt-1 font-semibold';
        statusElement.textContent = `Unavailable! The following dates are booked: ${bookedInRange.join(', ')}`;
        statusElement.parentElement.className = 'text-xs sm:text-sm text-red-800';
        statusElement.parentElement.parentElement.className = 'bg-red-50 border border-red-200 rounded p-3 sm:p-4 mb-4 sm:mb-6';
        submitButton.disabled = true;
        return false;
    }
    
    // All dates available
    const nights = datesInRange.length - 1;
    statusElement.textContent = `✓ Available! ${nights} night${nights !== 1 ? 's' : ''} selected.`;
    statusElement.parentElement.className = 'text-xs sm:text-sm text-green-800';
    statusElement.parentElement.parentElement.className = 'bg-green-50 border border-green-200 rounded p-3 sm:p-4 mb-4 sm:mb-6';
    submitButton.disabled = false;
    
    // Update pricing summary
    updatePricingSummary(nights);
    
    return true;
}

function updatePricingSummary(nights) {
    const pricePerNight = {{ $property->price_per_night }};
    const cautionFee = {{ $property->caution_fee }};
    
    if (nights > 0) {
        const subtotal = pricePerNight * nights;
        const total = subtotal + cautionFee;
        
        // Show nights breakdown
        document.getElementById('nights-count').textContent = nights;
        document.getElementById('nights-total').textContent = '₦' + subtotal.toLocaleString();
        document.getElementById('nights-display').classList.remove('hidden');
        
        // Show total
        document.getElementById('total-amount').textContent = '₦' + total.toLocaleString();
        document.getElementById('total-display').classList.remove('hidden');
        
        // Hide notice
        document.getElementById('select-dates-notice').classList.add('hidden');
    } else {
        // Hide pricing details
        document.getElementById('nights-display').classList.add('hidden');
        document.getElementById('total-display').classList.add('hidden');
        document.getElementById('select-dates-notice').classList.remove('hidden');
    }
}

function updateAvailabilityStatus() {
    if (bookedDates.length === 0) {
        document.getElementById('availability-status').textContent = 'All dates are available for booking.';
    } else {
        document.getElementById('availability-status').textContent = `${bookedDates.length} date(s) are already booked. Please select available dates.`;
    }
    
    // Validate current selection if dates are filled
    if (checkInInput.value && checkOutInput.value) {
        validateDates();
    }
}

// Event listeners
checkInInput.addEventListener('change', validateDates);
checkOutInput.addEventListener('change', validateDates);

// Discount code validation
function validateDiscountCode() {
    const code = document.getElementById('discount_code').value.trim();
    const messageEl = document.getElementById('discount-message');
    const detailsEl = document.getElementById('discount-details');
    
    if (!code) {
        messageEl.innerHTML = '<p class="text-red-600 text-xs sm:text-sm"><i class="fas fa-exclamation-circle mr-1"></i>Please enter a discount code</p>';
        detailsEl.classList.add('hidden');
        return;
    }
    
    messageEl.innerHTML = '<p class="text-gray-600 text-xs sm:text-sm"><i class="fas fa-spinner fa-spin mr-1"></i>Validating code...</p>';
    
    // Calculate total amount (this should match your booking calculation)
    const pricePerNight = {{ $property->price_per_night }};
    const cautionFee = {{ $property->caution_fee }};
    const checkIn = document.getElementById('check-in-date').value;
    const checkOut = document.getElementById('check-out-date').value;
    
    if (!checkIn || !checkOut) {
        messageEl.innerHTML = '<p class="text-red-600 text-xs sm:text-sm"><i class="fas fa-exclamation-circle mr-1"></i>Please select check-in and check-out dates first</p>';
        return;
    }
    
    const nights = Math.ceil((new Date(checkOut) - new Date(checkIn)) / (1000 * 60 * 60 * 24));
    const totalAmount = (pricePerNight * nights) + cautionFee;
    
    fetch('/api/v1/discount-codes/validate', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        credentials: 'same-origin',
        body: JSON.stringify({
            code: code,
            amount: totalAmount
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.valid) {
            messageEl.innerHTML = `<p class="text-green-600 text-xs sm:text-sm"><i class="fas fa-check-circle mr-1"></i>${data.message}</p>`;
            detailsEl.classList.remove('hidden');
            document.getElementById('discount-text').textContent = 
                `You save ₦${data.discount_amount.toLocaleString()}! New total: ₦${data.final_amount.toLocaleString()}`;
        } else {
            messageEl.innerHTML = `<p class="text-red-600 text-xs sm:text-sm"><i class="fas fa-times-circle mr-1"></i>${data.message}</p>`;
            detailsEl.classList.add('hidden');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        messageEl.innerHTML = '<p class="text-red-600 text-xs sm:text-sm"><i class="fas fa-exclamation-circle mr-1"></i>Error validating code. Please try again.</p>';
        detailsEl.classList.add('hidden');
    });
}

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    fetchBookedDates();
    submitButton.disabled = true; // Disable until dates are validated
});
</script>
@endsection