@extends('layouts.admin')

@section('title', 'Property Details - ' . $property->title)

@section('content')
<div class="max-w-6xl mx-auto">
    <div class="mb-6">
        <a href="{{ route('admin.properties.index') }}" class="text-primary hover:underline">
            <i class="fas fa-arrow-left mr-1"></i> Back to Properties
        </a>
    </div>

    <!-- Property Header -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <div class="flex justify-between items-start mb-4">
            <div>
                <h1 class="text-3xl font-bold mb-2">{{ $property->title }}</h1>
                <p class="text-gray-600">
                    <i class="fas fa-map-marker-alt mr-1"></i>
                    {{ $property->address }}, {{ $property->city }}, {{ $property->state }}
                </p>
                <p class="text-gray-600 mt-2">
                    <strong>Vendor:</strong> {{ $property->vendor->company_name }} 
                    ({{ $property->vendor->user->name }} - {{ $property->vendor->user->email }})
                </p>
            </div>
            <span class="px-4 py-2 rounded-full text-sm font-semibold
                {{ $property->status === 'pending' ? 'bg-yellow-100 text-yellow-800' : 
                   ($property->status === 'active' ? 'bg-green-100 text-green-800' : 
                   ($property->status === 'rejected' ? 'bg-red-100 text-red-800' : 'bg-gray-100 text-secondary')) }}">
                {{ ucfirst($property->status) }}
            </span>
        </div>

        @if($property->rejection_reason)
            <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-4">
                <p class="font-semibold text-red-800">Rejection Reason:</p>
                <p class="text-red-700">{{ $property->rejection_reason }}</p>
            </div>
        @endif

        <!-- Action Buttons -->
        <div class="flex gap-3 mt-4">
            @if($property->status === 'pending')
                <form method="POST" action="{{ route('admin.properties.approve', $property) }}">
                    @csrf
                    <button type="submit" 
                        class="bg-green-600 text-white px-6 py-3 rounded-lg hover:bg-green-700 font-semibold">
                        <i class="fas fa-check-circle mr-2"></i> Approve Property
                    </button>
                </form>
                
                <button onclick="openRejectModal()" 
                    class="bg-red-600 text-white px-6 py-3 rounded-lg hover:bg-red-700 font-semibold">
                    <i class="fas fa-times-circle mr-2"></i> Reject Property
                </button>
            @elseif($property->status === 'active' || $property->status === 'inactive')
                <form method="POST" action="{{ route('admin.properties.toggle', $property) }}">
                    @csrf
                    <button type="submit" 
                        class="bg-gray-600 text-white px-6 py-3 rounded-lg hover:bg-gray-700 font-semibold">
                        <i class="fas fa-power-off mr-2"></i>
                        {{ $property->status === 'active' ? 'Deactivate' : 'Activate' }} Property
                    </button>
                </form>
            @endif

            <a href="{{ route('properties.show', $property) }}" target="_blank"
                class="bg-primary text-white px-6 py-3 rounded-lg hover:bg-primary-dark font-semibold">
                <i class="fas fa-external-link-alt mr-2"></i> View Public Page
            </a>
        </div>
    </div>

    <!-- Property Photos -->
    @if($property->photos && count($property->photos) > 0)
        <div class="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 class="font-bold text-xl mb-4">Property Photos</h2>
            <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                @foreach($property->photos as $photo)
                    <img src="{{ asset('storage/' . $photo) }}" alt="{{ $property->title }}" 
                        class="w-full h-48 object-cover rounded-lg hover:opacity-90 cursor-pointer"
                        onclick="window.open('{{ asset('storage/' . $photo) }}', '_blank')">
                @endforeach
            </div>
        </div>
    @endif

    <!-- Property Details Grid -->
    <div class="grid md:grid-cols-2 gap-6 mb-6">
        <!-- Basic Details -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h2 class="font-bold text-xl mb-4">
                <i class="fas fa-info-circle mr-2 text-primary"></i> Property Details
            </h2>
            <div class="space-y-2">
                <p><strong>House Type:</strong> {{ $property->house_type }}</p>
                <p><strong>Bedrooms:</strong> {{ $property->bedrooms }}</p>
                <p><strong>Beds:</strong> {{ $property->beds }}</p>
                <p><strong>Bathrooms:</strong> {{ $property->bathrooms }}</p>
                <p><strong>Max Guests:</strong> {{ $property->max_guests }}</p>
                <p><strong>Allow Parties:</strong> 
                    <span class="{{ $property->allow_parties ? 'text-green-600' : 'text-red-600' }}">
                        {{ $property->allow_parties ? 'Yes' : 'No' }}
                    </span>
                </p>
            </div>
        </div>

        <!-- Pricing -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h2 class="font-bold text-xl mb-4">
                <i class="fas fa-money-bill-wave mr-2 text-green-600"></i> Pricing
            </h2>
            <div class="space-y-2">
                <p><strong>Price per Night:</strong> 
                    <span class="text-2xl text-green-600 font-bold">₦{{ number_format($property->price_per_night) }}</span>
                </p>
                <p><strong>Caution Fee:</strong> ₦{{ number_format($property->caution_fee) }}</p>
            </div>
        </div>

        <!-- Check-in/Check-out -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h2 class="font-bold text-xl mb-4">
                <i class="fas fa-clock mr-2 text-primary"></i> Check-in/Check-out
            </h2>
            <div class="space-y-2">
                <p><strong>Check-in From:</strong> {{ $property->check_in_from }}</p>
                <p><strong>Check-in Until:</strong> {{ $property->check_in_until }}</p>
                <p><strong>Check-out Time:</strong> {{ $property->check_out_time }}</p>
            </div>
        </div>

        <!-- Location -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h2 class="font-bold text-xl mb-4">
                <i class="fas fa-map-marker-alt mr-2 text-red-600"></i> Location
            </h2>
            <div class="space-y-2">
                <p><strong>Address:</strong> {{ $property->address }}</p>
                <p><strong>City:</strong> {{ $property->city }}</p>
                <p><strong>State:</strong> {{ $property->state }}</p>
                @if($property->location_lat && $property->location_lng)
                    <p><strong>Coordinates:</strong> {{ $property->location_lat }}, {{ $property->location_lng }}</p>
                @endif
            </div>
        </div>
    </div>

    <!-- Description -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <h2 class="font-bold text-xl mb-4">
            <i class="fas fa-align-left mr-2 text-gray-600"></i> Description
        </h2>
        <p class="text-gray-700 whitespace-pre-line">{{ $property->description }}</p>
    </div>

    <!-- Amenities -->
    @if($property->amenities && count($property->amenities) > 0)
        <div class="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 class="font-bold text-xl mb-4">
                <i class="fas fa-star mr-2 text-yellow-600"></i> Amenities
            </h2>
            <div class="grid grid-cols-2 md:grid-cols-4 gap-3">
                @foreach($property->amenities as $category => $items)
                    @if(is_array($items))
                        @foreach($items as $amenity)
                            <div class="flex items-center">
                                <i class="fas fa-check-circle text-green-600 mr-2"></i>
                                <span>{{ $amenity }}</span>
                            </div>
                        @endforeach
                    @endif
                @endforeach
            </div>
        </div>
    @endif

    <!-- Highlighted Features -->
    @if($property->highlighted_features && count($property->highlighted_features) > 0)
        <div class="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 class="font-bold text-xl mb-4">
                <i class="fas fa-gem mr-2 text-primary"></i> Highlighted Features
            </h2>
            <ul class="list-disc list-inside space-y-1">
                @foreach($property->highlighted_features as $feature)
                    <li>{{ $feature }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <!-- House Rules -->
    @if($property->house_rules && count($property->house_rules) > 0)
        <div class="bg-white rounded-lg shadow-md p-6 mb-6">
            <h2 class="font-bold text-xl mb-4">
                <i class="fas fa-list-ul mr-2 text-orange-600"></i> House Rules
            </h2>
            <ul class="list-disc list-inside space-y-1">
                @foreach($property->house_rules as $rule)
                    <li>{{ $rule }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <!-- Booking Statistics -->
    <div class="bg-white rounded-lg shadow-md p-6">
        <h2 class="font-bold text-xl mb-4">
            <i class="fas fa-chart-bar mr-2 text-indigo-600"></i> Booking Statistics
        </h2>
        <div class="grid md:grid-cols-4 gap-6">
            <div class="text-center">
                <p class="text-gray-600 mb-2">Total Bookings</p>
                <p class="text-3xl font-bold text-primary">{{ $property->bookings->count() }}</p>
            </div>
            <div class="text-center">
                <p class="text-gray-600 mb-2">Confirmed</p>
                <p class="text-3xl font-bold text-green-600">
                    {{ $property->bookings->where('booking_status', 'confirmed')->count() }}
                </p>
            </div>
            <div class="text-center">
                <p class="text-gray-600 mb-2">Pending</p>
                <p class="text-3xl font-bold text-yellow-600">
                    {{ $property->bookings->where('booking_status', 'pending')->count() }}
                </p>
            </div>
            <div class="text-center">
                <p class="text-gray-600 mb-2">Average Rating</p>
                <p class="text-3xl font-bold text-yellow-500">
                    {{ $property->averageRating() ? number_format($property->averageRating(), 1) . ' ★' : 'N/A' }}
                </p>
            </div>
        </div>
    </div>
</div>

<!-- Reject Modal -->
<div id="reject-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-8 max-w-md w-full mx-4">
        <h2 class="text-2xl font-bold mb-4">Reject Property</h2>
        
        <form method="POST" action="{{ route('admin.properties.reject', $property) }}">
            @csrf
            
            <div class="mb-4">
                <label class="block text-gray-700 mb-2 font-semibold">Reason for Rejection *</label>
                <textarea name="rejection_reason" rows="5" required
                    placeholder="Please provide a detailed reason for rejecting this property..."
                    class="w-full border-2 rounded-lg px-4 py-3 focus:border-red-500 focus:outline-none"></textarea>
                <p class="text-sm text-gray-600 mt-1">This will be sent to the vendor.</p>
            </div>

            <div class="flex gap-3">
                <button type="button" 
                    onclick="closeRejectModal()"
                    class="flex-1 bg-gray-200 text-gray-700 px-4 py-3 rounded-lg hover:bg-gray-300 font-semibold">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 bg-red-600 text-white px-4 py-3 rounded-lg hover:bg-red-700 font-semibold">
                    Reject Property
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openRejectModal() {
    document.getElementById('reject-modal').classList.remove('hidden');
}

function closeRejectModal() {
    document.getElementById('reject-modal').classList.add('hidden');
}

// Close modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeRejectModal();
    }
});

// Close modal when clicking outside
document.getElementById('reject-modal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeRejectModal();
    }
});
</script>
@endsection
