@extends('layouts.admin')

@section('title', 'Advertisement Plans')

@section('content')
<div>
    <div class="flex justify-between items-center mb-6">
        <h1 class="text-3xl font-bold">Advertisement Plans</h1>
        <button onclick="document.getElementById('create-modal').classList.remove('hidden')"
            class="bg-primary text-white px-6 py-3 rounded hover:bg-primary-dark font-semibold">
            <i class="fas fa-plus mr-2"></i> Create New Plan
        </button>
    </div>

    <!-- Placement Type Tabs -->
    <div class="mb-6 flex flex-wrap gap-3">
        <button onclick="filterByPlacement('all')" 
            class="placement-filter px-6 py-2 rounded-lg font-semibold transition active"
            data-type="all">
            <i class="fas fa-th mr-2"></i>All Plans
        </button>
        <button onclick="filterByPlacement('featured')" 
            class="placement-filter px-6 py-2 rounded-lg font-semibold transition"
            data-type="featured">
            <i class="fas fa-star mr-2"></i>Featured Listings
        </button>
        <button onclick="filterByPlacement('weekend_deals')" 
            class="placement-filter px-6 py-2 rounded-lg font-semibold transition"
            data-type="weekend_deals">
            <i class="fas fa-calendar-week mr-2"></i>Weekend Deals
        </button>
        <button onclick="filterByPlacement('top_unique')" 
            class="placement-filter px-6 py-2 rounded-lg font-semibold transition"
            data-type="top_unique">
            <i class="fas fa-gem mr-2"></i>Top Unique
        </button>
    </div>

    <!-- Ad Plans List -->
    <div class="grid md:grid-cols-3 gap-6">
        @forelse($adPlans as $plan)
            <div class="bg-white rounded-lg shadow-md p-6 plan-card {{ !$plan->active ? 'opacity-60' : '' }}" 
                data-placement="{{ $plan->placement_type }}">
                <div class="flex justify-between items-start mb-4">
                    <div>
                        <h3 class="text-2xl font-bold text-secondary">{{ $plan->name }}</h3>
                        <div class="flex gap-2 mt-2">
                            @if(!$plan->active)
                                <span class="inline-block bg-gray-200 text-gray-600 text-xs px-2 py-1 rounded">
                                    Inactive
                                </span>
                            @else
                                <span class="inline-block bg-green-100 text-green-800 text-xs px-2 py-1 rounded">
                                    Active
                                </span>
                            @endif
                            <span class="inline-block text-xs px-2 py-1 rounded
                                {{ $plan->placement_type === 'featured' ? 'bg-blue-100 text-blue-800' : ($plan->placement_type === 'weekend_deals' ? 'bg-orange-100 text-orange-800' : 'bg-purple-100 text-purple-800') }}">
                                <i class="fas {{ $plan->placement_type === 'featured' ? 'fa-star' : ($plan->placement_type === 'weekend_deals' ? 'fa-calendar-week' : 'fa-gem') }} mr-1"></i>
                                {{ ucfirst(str_replace('_', ' ', $plan->placement_type)) }}
                            </span>
                        </div>
                    </div>
                    <div class="flex gap-2">
                        <button onclick="openEditModal({{ $plan->id }}, '{{ $plan->name }}', {{ $plan->duration_days }}, {{ $plan->price }}, '{{ addslashes($plan->description ?? '') }}', {{ $plan->active ? 'true' : 'false' }}, '{{ $plan->placement_type }}')"
                            class="text-primary hover:text-blue-800">
                            <i class="fas fa-edit"></i>
                        </button>
                        @if(!$plan->vendorAds()->exists())
                            <form method="POST" action="{{ route('admin.ad-plans.destroy', $plan) }}" 
                                onsubmit="return confirm('Are you sure you want to delete this plan?')">
                                @csrf
                                @method('DELETE')
                                <button type="submit" class="text-red-600 hover:text-red-800">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </form>
                        @endif
                    </div>
                </div>

                <div class="mb-4">
                    <div class="text-4xl font-bold text-primary mb-2">
                        ₦{{ number_format($plan->price) }}
                    </div>
                    <div class="text-gray-600">
                        <i class="far fa-clock mr-1"></i> {{ $plan->duration_days }} days
                    </div>
                </div>

                @if($plan->description)
                    <p class="text-gray-600 text-sm mb-4">{{ $plan->description }}</p>
                @endif

                <div class="border-t pt-4">
                    <div class="text-sm text-gray-600">
                        <div class="flex justify-between mb-2">
                            <span>Active Ads:</span>
                            <span class="font-semibold">{{ $plan->vendorAds()->active()->count() }}</span>
                        </div>
                        <div class="flex justify-between">
                            <span>Total Sold:</span>
                            <span class="font-semibold">{{ $plan->vendorAds()->count() }}</span>
                        </div>
                    </div>
                </div>

                <div class="mt-4">
                    <form method="POST" action="{{ route('admin.ad-plans.toggle', $plan) }}">
                        @csrf
                        <button type="submit" 
                            class="w-full {{ $plan->active ? 'bg-gray-200 text-gray-700 hover:bg-gray-300' : 'bg-green-600 text-white hover:bg-green-700' }} px-4 py-2 rounded font-semibold">
                            {{ $plan->active ? 'Deactivate' : 'Activate' }}
                        </button>
                    </form>
                </div>
            </div>
        @empty
            <div class="col-span-3 bg-white rounded-lg shadow-md p-12 text-center">
                <i class="fas fa-ad text-gray-300 text-6xl mb-4"></i>
                <p class="text-gray-600 text-lg mb-4">No advertisement plans created yet</p>
                <button onclick="document.getElementById('create-modal').classList.remove('hidden')"
                    class="bg-primary text-white px-6 py-3 rounded hover:bg-primary-dark font-semibold">
                    Create Your First Plan
                </button>
            </div>
        @endforelse
    </div>
</div>

<!-- Create Modal -->
<div id="create-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-8 max-w-md w-full mx-4">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-2xl font-bold">Create Ad Plan</h2>
            <button onclick="document.getElementById('create-modal').classList.add('hidden')" 
                class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>

        <form method="POST" action="{{ route('admin.ad-plans.store') }}">
            @csrf

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Plan Name *</label>
                <input type="text" name="name" required
                    placeholder="e.g., Bronze, Silver, Gold"
                    class="w-full border rounded px-4 py-2 @error('name') border-red-500 @enderror">
                @error('name')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Placement Type *</label>
                <select name="placement_type" required
                    class="w-full border rounded px-4 py-2 @error('placement_type') border-red-500 @enderror">
                    <option value="">Select placement type...</option>
                    <option value="featured">Featured Listings</option>
                    <option value="weekend_deals">Weekend Deals</option>
                    <option value="top_unique">Top Unique Properties</option>
                </select>
                @error('placement_type')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Duration (Days) *</label>
                <input type="number" name="duration_days" min="1" max="365" required
                    placeholder="e.g., 30"
                    class="w-full border rounded px-4 py-2 @error('duration_days') border-red-500 @enderror">
                @error('duration_days')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Price (₦) *</label>
                <input type="number" name="price" min="0" step="0.01" required
                    placeholder="e.g., 5000"
                    class="w-full border rounded px-4 py-2 @error('price') border-red-500 @enderror">
                @error('price')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-6">
                <label class="block text-gray-700 mb-2">Description</label>
                <textarea name="description" rows="3"
                    placeholder="Brief description of this plan"
                    class="w-full border rounded px-4 py-2 @error('description') border-red-500 @enderror"></textarea>
                @error('description')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="flex gap-3">
                <button type="button" 
                    onclick="document.getElementById('create-modal').classList.add('hidden')"
                    class="flex-1 bg-gray-200 text-gray-700 px-4 py-3 rounded hover:bg-gray-300 font-semibold">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 bg-primary text-white px-4 py-3 rounded hover:bg-primary-dark font-semibold">
                    Create Plan
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Modal -->
<div id="edit-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-8 max-w-md w-full mx-4">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-2xl font-bold">Edit Ad Plan</h2>
            <button onclick="document.getElementById('edit-modal').classList.add('hidden')" 
                class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>

        <form id="edit-form" method="POST">
            @csrf
            @method('PUT')

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Plan Name *</label>
                <input type="text" id="edit-name" name="name" required
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Placement Type *</label>
                <select id="edit-placement-type" name="placement_type" required
                    class="w-full border rounded px-4 py-2">
                    <option value="featured">Featured Listings</option>
                    <option value="weekend_deals">Weekend Deals</option>
                    <option value="top_unique">Top Unique Properties</option>
                </select>
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Duration (Days) *</label>
                <input type="number" id="edit-duration" name="duration_days" min="1" max="365" required
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Price (₦) *</label>
                <input type="number" id="edit-price" name="price" min="0" step="0.01" required
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Description</label>
                <textarea id="edit-description" name="description" rows="3"
                    class="w-full border rounded px-4 py-2"></textarea>
            </div>

            <div class="mb-6">
                <label class="flex items-center">
                    <input type="checkbox" id="edit-active" name="active" value="1" class="mr-2">
                    <span class="text-gray-700">Active</span>
                </label>
            </div>

            <div class="flex gap-3">
                <button type="button" 
                    onclick="document.getElementById('edit-modal').classList.add('hidden')"
                    class="flex-1 bg-gray-200 text-gray-700 px-4 py-3 rounded hover:bg-gray-300 font-semibold">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 bg-primary text-white px-4 py-3 rounded hover:bg-primary-dark font-semibold">
                    Update Plan
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openEditModal(id, name, duration, price, description, active, placementType) {
    document.getElementById('edit-form').action = `/admin/ad-plans/${id}`;
    document.getElementById('edit-name').value = name;
    document.getElementById('edit-duration').value = duration;
    document.getElementById('edit-price').value = price;
    document.getElementById('edit-description').value = description;
    document.getElementById('edit-active').checked = active;
    document.getElementById('edit-placement-type').value = placementType;
    document.getElementById('edit-modal').classList.remove('hidden');
}

function filterByPlacement(type) {
    // Update filter buttons
    document.querySelectorAll('.placement-filter').forEach(btn => {
        if (btn.dataset.type === type) {
            btn.classList.add('active', 'bg-primary', 'text-white');
            btn.classList.remove('bg-gray-200', 'text-gray-700');
        } else {
            btn.classList.remove('active', 'bg-primary', 'text-white');
            btn.classList.add('bg-gray-200', 'text-gray-700');
        }
    });
    
    // Filter plan cards
    document.querySelectorAll('.plan-card').forEach(card => {
        if (type === 'all' || card.dataset.placement === type) {
            card.style.display = 'block';
        } else {
            card.style.display = 'none';
        }
    });
}

// Initialize filters
document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('.placement-filter').forEach(btn => {
        if (btn.classList.contains('active')) {
            btn.classList.add('bg-primary', 'text-white');
        } else {
            btn.classList.add('bg-gray-200', 'text-gray-700');
        }
    });
});
</script>

<style>
.placement-filter {
    cursor: pointer;
}
</style>
@endsection