<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('discount_deals', function (Blueprint $table) {
            $table->id();
            $table->string('code')->unique();
            $table->string('name');
            $table->text('description')->nullable();
            $table->enum('discount_type', ['percentage', 'fixed']); // percentage off or fixed amount off
            $table->decimal('discount_value', 10, 2); // % or amount
            $table->decimal('max_discount_amount', 10, 2)->nullable(); // max discount for percentage type
            $table->decimal('min_booking_amount', 10, 2)->default(0); // minimum booking amount to use deal
            $table->integer('usage_limit')->nullable(); // null = unlimited
            $table->integer('usage_count')->default(0);
            $table->integer('usage_per_user')->default(1); // how many times each user can use
            $table->timestamp('valid_from')->nullable();
            $table->timestamp('valid_until')->nullable();
            $table->boolean('is_active')->default(true);
            $table->boolean('is_featured')->default(false); // show on homepage
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->timestamps();
            
            $table->index('code');
            $table->index('is_active');
            $table->index('is_featured');
            $table->index(['valid_from', 'valid_until']);
        });
        
        // Track discount deal usage by users
        Schema::create('discount_deal_usage', function (Blueprint $table) {
            $table->id();
            $table->foreignId('discount_deal_id')->constrained()->onDelete('cascade');
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('booking_id')->constrained()->onDelete('cascade');
            $table->decimal('original_amount', 10, 2);
            $table->decimal('discount_amount', 10, 2);
            $table->decimal('final_amount', 10, 2);
            $table->timestamp('used_at');
            
            $table->unique(['discount_deal_id', 'booking_id']);
        });
        
        // Add discount fields to bookings table
        Schema::table('bookings', function (Blueprint $table) {
            $table->foreignId('discount_deal_id')->nullable()->after('vendor_payout_amount')->constrained()->onDelete('set null');
            $table->string('discount_code')->nullable()->after('discount_deal_id');
            $table->decimal('discount_amount', 10, 2)->default(0)->after('discount_code');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('bookings', function (Blueprint $table) {
            $table->dropForeign(['discount_deal_id']);
            $table->dropColumn(['discount_deal_id', 'discount_code', 'discount_amount']);
        });
        
        Schema::dropIfExists('discount_deal_usage');
        Schema::dropIfExists('discount_deals');
    }
};
