<?php

namespace App\Services;

use App\Models\PromoCode;
use App\Models\Vendor;
use App\Models\VendorAd;
use Illuminate\Support\Facades\DB;

class PromoCodeService
{
    public function validateCode(string $code, Vendor $vendor, float $amount): array
    {
        $promoCode = PromoCode::where('code', strtoupper($code))->first();

        if (!$promoCode) {
            return [
                'valid' => false,
                'message' => 'Invalid promo code.'
            ];
        }

        if (!$promoCode->isValid()) {
            return [
                'valid' => false,
                'message' => 'This promo code is not currently active or has expired.'
            ];
        }

        if (!$promoCode->canBeUsedBy($vendor)) {
            return [
                'valid' => false,
                'message' => 'You have already used this promo code the maximum number of times.'
            ];
        }

        if ($amount < $promoCode->min_ad_amount) {
            return [
                'valid' => false,
                'message' => "Minimum ad amount of ₦" . number_format($promoCode->min_ad_amount) . " required to use this code."
            ];
        }

        $discountAmount = $promoCode->calculateDiscount($amount);

        return [
            'valid' => true,
            'promo_code' => $promoCode,
            'discount_amount' => $discountAmount,
            'final_amount' => $amount - $discountAmount,
            'message' => 'Promo code applied successfully!'
        ];
    }

    public function applyPromoCode(PromoCode $promoCode, Vendor $vendor, VendorAd $vendorAd, float $originalAmount, float $discountAmount, float $finalAmount): void
    {
        DB::transaction(function () use ($promoCode, $vendor, $vendorAd, $originalAmount, $discountAmount, $finalAmount) {
            // Record usage
            DB::table('promo_code_usage')->insert([
                'promo_code_id' => $promoCode->id,
                'vendor_id' => $vendor->id,
                'vendor_ad_id' => $vendorAd->id,
                'original_amount' => $originalAmount,
                'discount_amount' => $discountAmount,
                'final_amount' => $finalAmount,
                'used_at' => now(),
            ]);

            // Increment usage count
            $promoCode->increment('usage_count');
        });
    }
}
