<?php

namespace App\Services\Payment;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PaystackService implements PaymentServiceInterface
{
    protected ?string $secretKey;
    protected ?string $publicKey;
    protected string $baseUrl = 'https://api.paystack.co';
    
    public function __construct()
    {
        $this->secretKey = config('services.paystack.secret_key');
        $this->publicKey = config('services.paystack.public_key');
    }
    
    protected function isConfigured(): bool
    {
        return !empty($this->secretKey) && !empty($this->publicKey);
    }
    
    public function initializePayment(array $data): array
    {
        if (!$this->isConfigured()) {
            Log::warning('Paystack is not configured');
            throw new \Exception('Payment service is not configured. Please contact support.');
        }
        
        $response = Http::withToken($this->secretKey)
            ->post("{$this->baseUrl}/transaction/initialize", [
                'email' => $data['email'],
                'amount' => $data['amount'] * 100, // Convert to kobo
                'reference' => $data['reference'],
                'callback_url' => $data['callback_url'] ?? config('app.url') . '/payment/callback',
                'metadata' => $data['metadata'] ?? [],
            ]);
            
        if (!$response->successful()) {
            Log::error('Paystack initialization failed', [
                'response' => $response->json(),
                'data' => $data,
            ]);
            
            throw new \Exception('Payment initialization failed');
        }
        
        $result = $response->json();
        
        return [
            'status' => 'success',
            'payment_url' => $result['data']['authorization_url'],
            'access_code' => $result['data']['access_code'],
            'reference' => $result['data']['reference'],
        ];
    }
    
    public function verifyPayment(string $reference): array
    {
        if (!$this->isConfigured()) {
            Log::warning('Paystack is not configured');
            throw new \Exception('Payment service is not configured. Please contact support.');
        }
        
        $response = Http::withToken($this->secretKey)
            ->get("{$this->baseUrl}/transaction/verify/{$reference}");
            
        if (!$response->successful()) {
            Log::error('Paystack verification failed', [
                'reference' => $reference,
                'response' => $response->json(),
            ]);
            
            throw new \Exception('Payment verification failed');
        }
        
        $result = $response->json();
        $data = $result['data'];
        
        return [
            'status' => $data['status'],
            'reference' => $data['reference'],
            'amount' => $data['amount'] / 100, // Convert from kobo
            'currency' => $data['currency'],
            'paid_at' => $data['paid_at'],
            'customer' => [
                'email' => $data['customer']['email'],
                'phone' => $data['customer']['phone'] ?? null,
            ],
            'metadata' => $data['metadata'] ?? [],
            'gateway_response' => $result,
        ];
    }
    
    public function verifyWebhook(array $payload, string $signature): bool
    {
        $secret = config('services.paystack.webhook_secret');
        $computedSignature = hash_hmac('sha512', json_encode($payload), $secret);
        
        return hash_equals($computedSignature, $signature);
    }
}