<?php

namespace App\Services\Notification;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class TwilioWhatsAppService implements WhatsAppServiceInterface
{
    protected string $sid;
    protected string $token;
    protected string $fromNumber;
    
    public function __construct()
    {
        $this->sid = config('services.twilio.sid');
        $this->token = config('services.twilio.token');
        $this->fromNumber = config('services.twilio.whatsapp_number');
    }
    
    public function sendMessage(string $to, string $message): bool
    {
        try {
            // Format phone number for WhatsApp
            $to = $this->formatPhoneNumber($to);
            
            $response = Http::asForm()
                ->withBasicAuth($this->sid, $this->token)
                ->post("https://api.twilio.com/2010-04-01/Accounts/{$this->sid}/Messages.json", [
                    'From' => $this->fromNumber,
                    'To' => "whatsapp:{$to}",
                    'Body' => $message,
                ]);
                
            if (!$response->successful()) {
                Log::error('Twilio WhatsApp failed', [
                    'to' => $to,
                    'response' => $response->json(),
                ]);
                return false;
            }
            
            return true;
        } catch (\Exception $e) {
            Log::error('Twilio WhatsApp exception', [
                'to' => $to,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }
    
    public function sendTemplate(string $to, string $templateName, array $params): bool
    {
        $message = $this->buildMessageFromTemplate($templateName, $params);
        return $this->sendMessage($to, $message);
    }
    
    protected function formatPhoneNumber(string $phone): string
    {
        // Remove any non-digit characters
        $phone = preg_replace('/\D/', '', $phone);
        
        // Add country code if not present (assuming Nigeria +234)
        if (!str_starts_with($phone, '234')) {
            if (str_starts_with($phone, '0')) {
                $phone = '234' . substr($phone, 1);
            } else {
                $phone = '234' . $phone;
            }
        }
        
        return '+' . $phone;
    }
    
    protected function buildMessageFromTemplate(string $templateName, array $params): string
    {
        return match($templateName) {
            'verification' => "🏨 ShortStayNG\n\nYour verification code is: {$params['code']}\n\nThis code expires in 10 minutes.\n\nIf you didn't request this, please ignore.",
            'booking_created' => "🏨 ShortStayNG\n\nNew booking request for {$params['property_name']}!\n\nGuests: {$params['guest_count']}\nDates: {$params['dates']}\n\nPlease review and respond.",
            'booking_approved' => "🏨 ShortStayNG\n\nYour booking for {$params['property_name']} has been approved!\n\nPlease complete payment: {$params['payment_link']}",
            'payment_confirmed' => "🏨 ShortStayNG\n\n✅ Payment confirmed!\n\nBooking ref: {$params['reference']}\nProperty: {$params['property_name']}\n\nContact details unlocked in your booking.",
            default => "ShortStayNG: " . ($params['message'] ?? 'Notification'),
        };
    }
}