<?php

namespace App\Services\Maps;

use App\Models\NearbyAttractionsCache;
use App\Models\Property;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class PlacesService
{
    protected ?string $apiKey;
    protected string $baseUrl = 'https://maps.googleapis.com/maps/api/place';
    
    public function __construct()
    {
        $this->apiKey = config('services.google.places_api_key');
    }
    
    protected function isConfigured(): bool
    {
        return !empty($this->apiKey);
    }
    
    public function getNearbyAttractions(Property $property, int $radius = 2000): array
    {
        // Check if API is configured
        if (!$this->isConfigured()) {
            Log::warning('Google Places API key not configured');
            return [];
        }
        
        // Check if property has coordinates
        if (!$property->location_lat || !$property->location_lng) {
            Log::warning('Property does not have coordinates', [
                'property_id' => $property->id,
            ]);
            return [];
        }
        
        // Check cache first
        $cached = $property->nearbyAttractionsCache()
            ->valid()
            ->first();
            
        if ($cached && !$cached->isExpired()) {
            return $cached->attractions;
        }
        
        // Fetch from API
        $attractions = $this->fetchNearbyPlaces(
            $property->location_lat,
            $property->location_lng,
            $radius
        );
        
        // Cache results for 24 hours
        NearbyAttractionsCache::updateOrCreate(
            ['property_id' => $property->id],
            [
                'latitude' => $property->location_lat,
                'longitude' => $property->location_lng,
                'attractions' => $attractions,
                'cached_at' => now(),
                'expires_at' => now()->addHours(24),
            ]
        );
        
        return $attractions;
    }
    
    protected function fetchNearbyPlaces(float $latitude, float $longitude, int $radius): array
    {
        try {
            $response = Http::get("{$this->baseUrl}/nearbysearch/json", [
                'location' => "{$latitude},{$longitude}",
                'radius' => $radius,
                'type' => 'tourist_attraction|point_of_interest|park|restaurant|shopping_mall',
                'key' => $this->apiKey,
            ]);
            
            if (!$response->successful()) {
                Log::error('Google Places API failed', [
                    'coordinates' => "{$latitude},{$longitude}",
                    'response' => $response->json(),
                ]);
                return [];
            }
            
            $data = $response->json();
            
            if ($data['status'] !== 'OK' || empty($data['results'])) {
                return [];
            }
            
            $attractions = [];
            
            foreach (array_slice($data['results'], 0, 10) as $place) {
                $attraction = [
                    'name' => $place['name'],
                    'vicinity' => $place['vicinity'] ?? null,
                    'rating' => $place['rating'] ?? null,
                    'types' => $place['types'] ?? [],
                    'photo_url' => null,
                ];
                
                // Get photo if available
                if (!empty($place['photos'][0]['photo_reference'])) {
                    $attraction['photo_url'] = $this->getPhotoUrl($place['photos'][0]['photo_reference']);
                }
                
                // Get place details for description
                $details = $this->getPlaceDetails($place['place_id']);
                if ($details) {
                    $attraction['description'] = $details['description'] ?? null;
                    $attraction['website'] = $details['website'] ?? null;
                    $attraction['phone'] = $details['phone'] ?? null;
                }
                
                $attractions[] = $attraction;
            }
            
            return $attractions;
        } catch (\Exception $e) {
            Log::error('Google Places API exception', [
                'error' => $e->getMessage(),
            ]);
            return [];
        }
    }
    
    protected function getPlaceDetails(string $placeId): ?array
    {
        try {
            // Use cache to avoid repeated API calls
            return Cache::remember("place_details_{$placeId}", 86400, function () use ($placeId) {
                $response = Http::get("{$this->baseUrl}/details/json", [
                    'place_id' => $placeId,
                    'fields' => 'editorial_summary,website,formatted_phone_number',
                    'key' => $this->apiKey,
                ]);
                
                if (!$response->successful()) {
                    return null;
                }
                
                $data = $response->json();
                
                if ($data['status'] !== 'OK' || empty($data['result'])) {
                    return null;
                }
                
                $result = $data['result'];
                
                return [
                    'description' => $result['editorial_summary']['overview'] ?? null,
                    'website' => $result['website'] ?? null,
                    'phone' => $result['formatted_phone_number'] ?? null,
                ];
            });
        } catch (\Exception $e) {
            Log::error('Google Place Details exception', [
                'place_id' => $placeId,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }
    
    protected function getPhotoUrl(string $photoReference, int $maxWidth = 400): string
    {
        return "{$this->baseUrl}/photo?" . http_build_query([
            'photo_reference' => $photoReference,
            'maxwidth' => $maxWidth,
            'key' => $this->apiKey,
        ]);
    }
    
    public function searchPlaces(string $query, float $latitude, float $longitude, int $radius = 5000): array
    {
        if (!$this->isConfigured()) {
            Log::warning('Google Places API key not configured');
            return [];
        }
        
        try {
            $response = Http::get("{$this->baseUrl}/textsearch/json", [
                'query' => $query,
                'location' => "{$latitude},{$longitude}",
                'radius' => $radius,
                'key' => $this->apiKey,
            ]);
            
            if (!$response->successful()) {
                return [];
            }
            
            $data = $response->json();
            
            if ($data['status'] !== 'OK' || empty($data['results'])) {
                return [];
            }
            
            return array_map(function ($place) {
                return [
                    'place_id' => $place['place_id'],
                    'name' => $place['name'],
                    'address' => $place['formatted_address'] ?? null,
                    'rating' => $place['rating'] ?? null,
                ];
            }, $data['results']);
        } catch (\Exception $e) {
            Log::error('Google Places search exception', [
                'query' => $query,
                'error' => $e->getMessage(),
            ]);
            return [];
        }
    }
}