<?php

// app/Services/Maps/GoogleMapsService.php

namespace App\Services\Maps;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class GoogleMapsService
{
    protected ?string $apiKey;
    protected string $baseUrl = 'https://maps.googleapis.com/maps/api';
    
    public function __construct()
    {
        $this->apiKey = config('services.google.maps_api_key');
    }
    
    protected function isConfigured(): bool
    {
        return !empty($this->apiKey);
    }
    
    public function geocodeAddress(string $address): ?array
    {
        if (!$this->isConfigured()) {
            Log::warning('Google Maps API key not configured');
            return null;
        }
        
        try {
            $response = Http::get("{$this->baseUrl}/geocode/json", [
                'address' => $address,
                'key' => $this->apiKey,
            ]);
            
            if (!$response->successful()) {
                Log::error('Google Maps geocoding failed', [
                    'address' => $address,
                    'response' => $response->json(),
                ]);
                return null;
            }
            
            $data = $response->json();
            
            if ($data['status'] !== 'OK' || empty($data['results'])) {
                return null;
            }
            
            $result = $data['results'][0];
            $location = $result['geometry']['location'];
            
            return [
                'latitude' => $location['lat'],
                'longitude' => $location['lng'],
                'formatted_address' => $result['formatted_address'],
            ];
        } catch (\Exception $e) {
            Log::error('Google Maps geocoding exception', [
                'address' => $address,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }
    
    public function reverseGeocode(float $latitude, float $longitude): ?array
    {
        if (!$this->isConfigured()) {
            Log::warning('Google Maps API key not configured');
            return null;
        }
        
        try {
            $response = Http::get("{$this->baseUrl}/geocode/json", [
                'latlng' => "{$latitude},{$longitude}",
                'key' => $this->apiKey,
            ]);
            
            if (!$response->successful()) {
                Log::error('Google Maps reverse geocoding failed', [
                    'coordinates' => "{$latitude},{$longitude}",
                    'response' => $response->json(),
                ]);
                return null;
            }
            
            $data = $response->json();
            
            if ($data['status'] !== 'OK' || empty($data['results'])) {
                return null;
            }
            
            $result = $data['results'][0];
            
            return [
                'formatted_address' => $result['formatted_address'],
                'address_components' => $result['address_components'],
            ];
        } catch (\Exception $e) {
            Log::error('Google Maps reverse geocoding exception', [
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }
    
    public function getStaticMapUrl(float $latitude, float $longitude, int $zoom = 15, string $size = '600x400'): ?string
    {
        if (!$this->isConfigured()) {
            return null;
        }
        
        return "{$this->baseUrl}/staticmap?" . http_build_query([
            'center' => "{$latitude},{$longitude}",
            'zoom' => $zoom,
            'size' => $size,
            'markers' => "color:red|{$latitude},{$longitude}",
            'key' => $this->apiKey,
        ]);
    }
}
