<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ImageWatermarkService
{
    /**
     * Add watermark to an image using GD library
     * 
     * @param string $imagePath Path to the image in storage
     * @param string $position Position of watermark (bottom-right, bottom-left, top-right, top-left, center)
     * @param int $opacity Opacity of watermark (0-100)
     * @return string Path to watermarked image
     */
    public function addWatermark($imagePath, $position = 'bottom-right', $opacity = 50)
    {
        // Get the logo from settings or use default
        $logoPath = \App\Models\Setting::get('logo');
        
        if (!$logoPath) {
            // No logo configured, return original image
            return $imagePath;
        }

        $fullLogoPath = storage_path('app/public/' . $logoPath);
        
        if (!file_exists($fullLogoPath)) {
            // Logo file doesn't exist, return original image
            return $imagePath;
        }

        // Load the main image
        $fullImagePath = storage_path('app/public/' . $imagePath);
        
        if (!file_exists($fullImagePath)) {
            return $imagePath;
        }

        try {
            // Get image info
            $imageInfo = getimagesize($fullImagePath);
            $logoInfo = getimagesize($fullLogoPath);
            
            if (!$imageInfo || !$logoInfo) {
                return $imagePath;
            }
            
            // Create image resources
            $image = $this->createImageResource($fullImagePath, $imageInfo[2]);
            $logo = $this->createImageResource($fullLogoPath, $logoInfo[2]);
            
            if (!$image || !$logo) {
                return $imagePath;
            }
            
            // Calculate watermark size (15% of image width)
            $watermarkWidth = (int)($imageInfo[0] * 0.15);
            $watermarkHeight = (int)($logoInfo[1] * ($watermarkWidth / $logoInfo[0]));
            
            // Resize logo
            $resizedLogo = imagecreatetruecolor($watermarkWidth, $watermarkHeight);
            
            // Preserve transparency
            imagealphablending($resizedLogo, false);
            imagesavealpha($resizedLogo, true);
            $transparent = imagecolorallocatealpha($resizedLogo, 255, 255, 255, 127);
            imagefilledrectangle($resizedLogo, 0, 0, $watermarkWidth, $watermarkHeight, $transparent);
            
            // Resize the logo
            imagecopyresampled($resizedLogo, $logo, 0, 0, 0, 0, $watermarkWidth, $watermarkHeight, $logoInfo[0], $logoInfo[1]);
            
            // Calculate position
            $padding = 20;
            $x = 0;
            $y = 0;
            
            switch ($position) {
                case 'bottom-right':
                    $x = $imageInfo[0] - $watermarkWidth - $padding;
                    $y = $imageInfo[1] - $watermarkHeight - $padding;
                    break;
                case 'bottom-left':
                    $x = $padding;
                    $y = $imageInfo[1] - $watermarkHeight - $padding;
                    break;
                case 'top-right':
                    $x = $imageInfo[0] - $watermarkWidth - $padding;
                    $y = $padding;
                    break;
                case 'top-left':
                    $x = $padding;
                    $y = $padding;
                    break;
                case 'center':
                    $x = ($imageInfo[0] - $watermarkWidth) / 2;
                    $y = ($imageInfo[1] - $watermarkHeight) / 2;
                    break;
            }
            
            // Enable alpha blending for proper transparency
            imagealphablending($image, true);
            
            // Merge watermark with image (with opacity)
            imagecopymerge($image, $resizedLogo, $x, $y, 0, 0, $watermarkWidth, $watermarkHeight, $opacity);
            
            // Save the watermarked image
            $this->saveImage($image, $fullImagePath, $imageInfo[2]);
            
            // Free memory
            imagedestroy($image);
            imagedestroy($logo);
            imagedestroy($resizedLogo);
            
            return $imagePath;
            
        } catch (\Exception $e) {
            // If watermarking fails, return original image
            Log::error('Watermark failed: ' . $e->getMessage());
            return $imagePath;
        }
    }
    
    /**
     * Create image resource from file
     */
    private function createImageResource($path, $type)
    {
        switch ($type) {
            case IMAGETYPE_JPEG:
                return imagecreatefromjpeg($path);
            case IMAGETYPE_PNG:
                return imagecreatefrompng($path);
            case IMAGETYPE_GIF:
                return imagecreatefromgif($path);
            case IMAGETYPE_WEBP:
                return imagecreatefromwebp($path);
            default:
                return null;
        }
    }
    
    /**
     * Save image resource to file
     */
    private function saveImage($resource, $path, $type)
    {
        switch ($type) {
            case IMAGETYPE_JPEG:
                imagejpeg($resource, $path, 90);
                break;
            case IMAGETYPE_PNG:
                imagepng($resource, $path, 9);
                break;
            case IMAGETYPE_GIF:
                imagegif($resource, $path);
                break;
            case IMAGETYPE_WEBP:
                imagewebp($resource, $path, 90);
                break;
        }
    }
    
    /**
     * Add watermark to multiple images
     * 
     * @param array $imagePaths Array of image paths
     * @param string $position Position of watermark
     * @param int $opacity Opacity of watermark
     * @return array Array of watermarked image paths
     */
    public function addWatermarkToMultiple(array $imagePaths, $position = 'bottom-right', $opacity = 50)
    {
        $watermarkedPaths = [];
        
        foreach ($imagePaths as $imagePath) {
            $watermarkedPaths[] = $this->addWatermark($imagePath, $position, $opacity);
        }
        
        return $watermarkedPaths;
    }
}

