<?php

namespace App\Services;

use App\Models\DiscountDeal;
use App\Models\User;
use App\Models\Booking;
use Illuminate\Support\Facades\DB;

class DiscountDealService
{
    public function validateCode(string $code, User $user, float $amount): array
    {
        $discountDeal = DiscountDeal::where('code', strtoupper($code))->first();

        if (!$discountDeal) {
            return [
                'valid' => false,
                'message' => 'Invalid discount code.'
            ];
        }

        if (!$discountDeal->isValid()) {
            return [
                'valid' => false,
                'message' => 'This discount code is not currently active or has expired.'
            ];
        }

        if (!$discountDeal->canBeUsedBy($user)) {
            return [
                'valid' => false,
                'message' => 'You have already used this discount code the maximum number of times.'
            ];
        }

        if ($amount < $discountDeal->min_booking_amount) {
            return [
                'valid' => false,
                'message' => "Minimum booking amount of ₦" . number_format($discountDeal->min_booking_amount) . " required to use this code."
            ];
        }

        $discountAmount = $discountDeal->calculateDiscount($amount);

        return [
            'valid' => true,
            'discount_deal' => $discountDeal,
            'discount_amount' => $discountAmount,
            'final_amount' => $amount - $discountAmount,
            'message' => 'Discount code applied successfully!'
        ];
    }

    public function applyDiscountDeal(DiscountDeal $discountDeal, User $user, Booking $booking, float $originalAmount, float $discountAmount, float $finalAmount): void
    {
        DB::transaction(function () use ($discountDeal, $user, $booking, $originalAmount, $discountAmount, $finalAmount) {
            // Check if usage already exists (prevent duplicates)
            $existingUsage = DB::table('discount_deal_usage')
                ->where('discount_deal_id', $discountDeal->id)
                ->where('booking_id', $booking->id)
                ->exists();

            if (!$existingUsage) {
                // Record usage
                DB::table('discount_deal_usage')->insert([
                    'discount_deal_id' => $discountDeal->id,
                    'user_id' => $user->id,
                    'booking_id' => $booking->id,
                    'original_amount' => $originalAmount,
                    'discount_amount' => $discountAmount,
                    'final_amount' => $finalAmount,
                    'used_at' => now(),
                ]);

                // Increment usage count
                $discountDeal->increment('usage_count');
            }
        });
    }
}
