<?php

namespace App\Services;

use App\Models\Booking;
use App\Models\Property;
use App\Models\User;
use App\Models\Vendor;
use App\Events\BookingCreated;
use App\Events\BookingApproved;
use Illuminate\Support\Str;

class BookingService
{
    public function createBooking(User $user, Property $property, array $data): Booking
    {
        $startDate = \Carbon\Carbon::parse($data['start_date']);
        $endDate = \Carbon\Carbon::parse($data['end_date']);
        $nightsCount = $startDate->diffInDays($endDate);
        
        $guestCount = $data['adults_count'] + $data['children_count'];
        $tripType = $data['trip_type'] ?? 'normal';
        
        // Determine if vendor approval required
        $requiresApproval = $tripType === 'party' || $guestCount > $property->max_allowed_guests;
        
        // Calculate amounts
        $subtotal = $property->price_per_night * $nightsCount;
        $cautionFee = $property->caution_fee;
        $totalAmount = $subtotal + $cautionFee;
        
        // Apply discount deal if provided
        $discountDealId = null;
        $discountCode = null;
        $discountAmount = 0;
        
        if (!empty($data['discount_code'])) {
            $discountService = app(DiscountDealService::class);
            $validation = $discountService->validateCode($data['discount_code'], $user, $totalAmount);
            
            if ($validation['valid']) {
                $discountDealId = $validation['discount_deal']->id;
                $discountCode = $validation['discount_deal']->code;
                $discountAmount = $validation['discount_amount'];
                $totalAmount = $validation['final_amount'];
            }
        }
        
        // Calculate commission (ONLY on subtotal, NOT on caution fee since it's refundable)
        // No commission for company-owned properties
        $isCompanyProperty = $property->vendor->company_name === 'ShortStayNG';
        
        if ($isCompanyProperty) {
            $commissionRate = 0;
            $commissionAmount = 0;
            $vendorPayoutAmount = $subtotal;
        } else {
            $commissionRate = $property->vendor->commission_rate_at_signup;
            $commissionAmount = ($subtotal * $commissionRate) / 100;
            $vendorPayoutAmount = $subtotal - $commissionAmount;
        }
        
        $booking = Booking::create([
            'user_id' => $user->id,
            'property_id' => $property->id,
            'vendor_id' => $property->vendor_id,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'nights_count' => $nightsCount,
            'adults_count' => $data['adults_count'],
            'children_count' => $data['children_count'],
            'guest_count' => $guestCount,
            'trip_type' => $tripType,
            'requires_vendor_approval' => $requiresApproval,
            'vendor_approval_status' => $requiresApproval ? 'pending' : 'approved',
            'property_price_per_night' => $property->price_per_night,
            'subtotal' => $subtotal,
            'caution_fee_amount' => $cautionFee,
            'total_amount' => $totalAmount,
            'payment_reference' => 'BK-' . Str::upper(Str::random(12)),
            'commission_rate' => $commissionRate,
            'commission_amount' => $commissionAmount,
            'vendor_payout_amount' => $vendorPayoutAmount,
            'discount_deal_id' => $discountDealId,
            'discount_code' => $discountCode,
            'discount_amount' => $discountAmount,
        ]);
        
        // NOTE: Discount deal usage will be recorded AFTER payment is confirmed
        // See WebhookController and Admin/PaymentVerificationController for usage tracking
        
        event(new BookingCreated($booking));
        
        return $booking;
    }
    
    public function approveBooking(Booking $booking, Vendor $vendor): bool
    {
        if ($booking->vendor_id !== $vendor->id) {
            return false;
        }
        
        $booking->update([
            'vendor_approval_status' => 'approved',
            'vendor_approved_at' => now(),
        ]);
        
        event(new BookingApproved($booking));
        
        return true;
    }
    
    public function rejectBooking(Booking $booking, Vendor $vendor, string $reason): bool
    {
        if ($booking->vendor_id !== $vendor->id) {
            return false;
        }
        
        $booking->update([
            'vendor_approval_status' => 'rejected',
            'vendor_rejection_reason' => $reason,
            'booking_status' => 'cancelled',
            'cancelled_at' => now(),
        ]);
        
        return true;
    }
    
    public function verifyPayment(Booking $booking, User $admin): void
    {
        $booking->update([
            'admin_verified' => true,
            'verified_by' => $admin->id,
            'admin_verified_at' => now(),
            'booking_status' => 'confirmed',
            'user_visible_vendor_info' => true,
            'vendor_visible_user_info' => true,
            'map_visible' => true,
        ]);
    }
    
    public function completeBooking(Booking $booking): void
    {
        $booking->update([
            'booking_status' => 'completed',
        ]);
    }
    
    /**
     * Cancel conflicting unpaid bookings when a booking is paid
     * This ensures that if multiple users book the same dates,
     * only the first one to pay gets the booking (if no units available)
     */
    public function cancelConflictingUnpaidBookings(Booking $paidBooking): int
    {
        $property = $paidBooking->property;
        
        // Check if there are still available units
        $availableUnits = $property->getAvailableUnits($paidBooking->start_date, $paidBooking->end_date);
        
        // If there are available units, don't cancel other bookings
        if ($availableUnits > 0) {
            return 0;
        }
        
        // Find conflicting unpaid bookings for the same property and dates
        $conflictingBookings = Booking::where('property_id', $paidBooking->property_id)
            ->where('id', '!=', $paidBooking->id)
            ->whereIn('payment_status', ['pending', 'pending_verification'])
            ->whereNotIn('booking_status', ['cancelled', 'completed'])
            ->where(function ($query) use ($paidBooking) {
                $query->whereBetween('start_date', [$paidBooking->start_date, $paidBooking->end_date])
                    ->orWhereBetween('end_date', [$paidBooking->start_date, $paidBooking->end_date])
                    ->orWhere(function ($q) use ($paidBooking) {
                        $q->where('start_date', '<=', $paidBooking->start_date)
                          ->where('end_date', '>=', $paidBooking->end_date);
                    });
            })
            ->get();
        
        $cancelledCount = 0;
        
        foreach ($conflictingBookings as $booking) {
            $booking->update([
                'booking_status' => 'cancelled',
                'cancelled_at' => now(),
                'cancellation_reason' => 'Automatically cancelled: Property no longer available for selected dates. Another booking was confirmed.',
            ]);
            
            $cancelledCount++;
            
            // TODO: Send notification to user about cancellation
            \Log::info('Booking auto-cancelled due to conflict', [
                'cancelled_booking_id' => $booking->id,
                'user_id' => $booking->user_id,
                'paid_booking_id' => $paidBooking->id,
                'property_id' => $property->id,
            ]);
        }
        
        return $cancelledCount;
    }
}