<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class VendorAd extends Model
{
    use HasFactory;

    protected $fillable = [
        'vendor_id',
        'property_id',
        'ad_plan_id',
        'placement_type',
        'payment_reference',
        'payment_status',
        'payment_method',
        'payment_proof',
        'promo_code_id',
        'promo_code',
        'discount_amount',
        'final_amount',
        'approved_by_admin',
        'approved_by',
        'approved_at',
        'start_date',
        'end_date',
        'active',
    ];

    protected $casts = [
        'approved_by_admin' => 'boolean',
        'active' => 'boolean',
        'approved_at' => 'datetime',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'discount_amount' => 'decimal:2',
        'final_amount' => 'decimal:2',
    ];

    // Relationships
    public function vendor()
    {
        return $this->belongsTo(Vendor::class);
    }

    public function property()
    {
        return $this->belongsTo(Property::class);
    }

    public function adPlan()
    {
        return $this->belongsTo(AdPlan::class);
    }

    public function promoCode()
    {
        return $this->belongsTo(PromoCode::class);
    }

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('active', true)
            ->where('end_date', '>=', now());
    }

    public function scopePendingApproval($query)
    {
        return $query->where('payment_status', 'paid')
            ->where('approved_by_admin', false);
    }

    public function scopePendingPayment($query)
    {
        return $query->where('payment_status', 'pending');
    }

    public function scopeFeatured($query)
    {
        return $query->where('placement_type', 'featured');
    }

    public function scopeWeekendDeals($query)
    {
        return $query->where('placement_type', 'weekend_deals');
    }

    public function scopeTopUnique($query)
    {
        return $query->where('placement_type', 'top_unique');
    }

    // Accessors
    public function getPlacementTypeNameAttribute()
    {
        return match($this->placement_type) {
            'featured' => 'Featured Listings',
            'weekend_deals' => 'Weekend Deals',
            'top_unique' => 'Top Unique Properties',
            default => ucfirst(str_replace('_', ' ', $this->placement_type))
        };
    }

    public function getIsExpiredAttribute()
    {
        return $this->end_date && $this->end_date->isPast();
    }

    public function getDaysRemainingAttribute()
    {
        if (!$this->end_date || $this->is_expired) {
            return 0;
        }
        return now()->diffInDays($this->end_date);
    }
}