<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Vendor extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'company_name',
        'company_description',
        'profile_photo',
        'commission_rate_at_signup',
        'verified',
        'bank_name',
        'account_number',
        'account_name',
        'whatsapp_number',
        'transaction_pin',
        'pin_set_at',
        'payout_balance',
        'pending_balance',
        'total_earned',
        // KYC fields
        'id_card_type',
        'id_card_number',
        'id_card_document',
        'cac_document',
        'business_address',
        'home_address',
        'kyc_status',
        'kyc_rejection_reason',
        'kyc_submitted_at',
        'kyc_approved_at',
        // Car Rental fields
        'service_types',
        'car_rental_business_name',
        'car_rental_business_registration',
        'car_rental_business_certificate',
        'car_rental_license',
        'car_rental_insurance_certificate',
        'car_rental_business_address',
        'car_rental_tax_id',
        'car_rental_bank_name',
        'car_rental_account_number',
        'car_rental_account_name',
        'car_rental_kyc_status',
        'car_rental_rejection_reason',
        'car_rental_verified_at',
    ];

    protected $casts = [
        'commission_rate_at_signup' => 'decimal:2',
        'payout_balance' => 'decimal:2',
        'pending_balance' => 'decimal:2',
        'total_earned' => 'decimal:2',
        'verified' => 'boolean',
        'kyc_submitted_at' => 'datetime',
        'kyc_approved_at' => 'datetime',
        'pin_set_at' => 'datetime',
        'service_types' => 'array',
        'car_rental_verified_at' => 'datetime',
    ];

    protected $hidden = [
        'transaction_pin',
    ];
    
    public function isKycApproved()
    {
        return $this->kyc_status === 'approved';
    }
    
    public function isKycPending()
    {
        return $this->kyc_status === 'pending';
    }
    
    public function hasSubmittedKyc()
    {
        return !is_null($this->kyc_submitted_at);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function properties()
    {
        return $this->hasMany(Property::class);
    }

    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function ads()
    {
        return $this->hasMany(VendorAd::class);
    }

    public function payouts()
    {
        return $this->hasMany(Payout::class);
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    public function activeProperties()
    {
        return $this->properties()->where('status', 'active');
    }

    public function featuredProperties()
    {
        return $this->properties()
            ->where('status', 'active')
            ->whereNotNull('featured_until')
            ->where('featured_until', '>=', now());
    }

    // Car Rental Relationships
    public function cars()
    {
        return $this->hasMany(Car::class);
    }

    public function carBookings()
    {
        return $this->hasMany(CarBooking::class);
    }

    public function activeCars()
    {
        return $this->cars()->where('status', 'approved')->where('is_available', true);
    }

    // Service Type Helpers
    public function offersProperties()
    {
        return in_array('property', $this->service_types ?? []);
    }

    public function offersCarRentals()
    {
        return in_array('car_rental', $this->service_types ?? []);
    }

    public function isCarRentalKycApproved()
    {
        // If vendor offers car rentals and general KYC is approved, they can add cars
        return $this->offersCarRentals() && $this->kyc_status === 'approved';
    }

    public function isCarRentalKycPending()
    {
        return $this->car_rental_kyc_status === 'pending';
    }

    public function hasSubmittedCarRentalKyc()
    {
        return !is_null($this->car_rental_business_name);
    }
}
