<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PropertyRoom extends Model
{
    use HasFactory;

    protected $fillable = [
        'property_id',
        'room_type',
        'description',
        'price_per_night',
        'max_guests',
        'total_units',
        'bedrooms',
        'beds',
        'bathrooms',
        'room_size_sqm',
        'amenities',
        'photos',
        'status',
    ];

    protected $casts = [
        'price_per_night' => 'decimal:2',
        'room_size_sqm' => 'decimal:2',
        'amenities' => 'array',
        'photos' => 'array',
    ];

    // Relationships
    public function property()
    {
        return $this->belongsTo(Property::class);
    }

    public function bookings()
    {
        return $this->hasMany(Booking::class, 'room_id');
    }

    // Helper Methods
    public function getAvailableUnits($startDate, $endDate): int
    {
        $start = \Carbon\Carbon::parse($startDate);
        $end = \Carbon\Carbon::parse($endDate);
        
        $bookedUnits = $this->bookings()
            ->whereIn('booking_status', ['confirmed', 'checked_in'])
            ->where('payment_status', 'paid')
            ->where(function ($query) use ($start, $end) {
                $query->whereBetween('start_date', [$start, $end])
                    ->orWhereBetween('end_date', [$start, $end])
                    ->orWhere(function ($q) use ($start, $end) {
                        $q->where('start_date', '<=', $start)
                          ->where('end_date', '>=', $end);
                    });
            })
            ->count();
        
        return max(0, $this->total_units - $bookedUnits);
    }

    public function hasAvailableUnits($startDate, $endDate): bool
    {
        return $this->getAvailableUnits($startDate, $endDate) > 0;
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }
}