<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Property extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'vendor_id',
        'title',
        'description',
        'price_per_night',
        'caution_fee',
        'max_allowed_guests',
        'allow_parties',
        'featured_until',
        'ad_plan_id',
        'weekend_deals_until',
        'weekend_deals_plan_id',
        'top_unique_until',
        'top_unique_plan_id',
        'status',
        'address',
        'city',
        'state',
        'country',
        'location_lat',
        'location_lng',
        'map_visible_after_payment',
        'house_type',
        'is_hotel', // NEW
        'bedrooms',
        'beds',
        'bathrooms',
        'max_guests',
        'total_units',
        'highlighted_features',
        'amenities',
        'house_rules',
        'photos',
        'video_url', // NEW
        'check_in_from',
        'check_in_until',
        'check_out_time',
        'unavailable_dates',
        'cancellation_policy',
        'quality_standard',
        'verified',
        'quality_assured',
        'rejection_reason',
    ];

    protected $casts = [
        'price_per_night' => 'decimal:2',
        'caution_fee' => 'decimal:2',
        'location_lat' => 'decimal:8',
        'location_lng' => 'decimal:8',
        'allow_parties' => 'boolean',
        'is_hotel' => 'boolean', // NEW
        'map_visible_after_payment' => 'boolean',
        'quality_standard' => 'boolean',
        'verified' => 'boolean',
        'quality_assured' => 'boolean',
        'featured_until' => 'datetime',
        'weekend_deals_until' => 'datetime',
        'top_unique_until' => 'datetime',
        'highlighted_features' => 'array',
        'amenities' => 'array',
        'house_rules' => 'array',
        'photos' => 'array',
        'unavailable_dates' => 'array',
    ];

    // Relationships
    public function vendor()
    {
        return $this->belongsTo(Vendor::class);
    }

    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function rooms() // NEW
    {
        return $this->hasMany(PropertyRoom::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function adPlan()
    {
        return $this->belongsTo(AdPlan::class);
    }

    public function weekendDealsPlan()
    {
        return $this->belongsTo(AdPlan::class, 'weekend_deals_plan_id');
    }

    public function topUniquePlan()
    {
        return $this->belongsTo(AdPlan::class, 'top_unique_plan_id');
    }

    public function vendorAds()
    {
        return $this->hasMany(VendorAd::class);
    }

    public function nearbyAttractionsCache()
    {
        return $this->hasOne(NearbyAttractionsCache::class);
    }

    // Helper Methods
    public function isFeatured(): bool
    {
        return !is_null($this->featured_until) && $this->featured_until->isFuture();
    }

    public function isWeekendDeal(): bool
    {
        return !is_null($this->weekend_deals_until) && $this->weekend_deals_until->isFuture();
    }

    public function isTopUnique(): bool
    {
        return !is_null($this->top_unique_until) && $this->top_unique_until->isFuture();
    }

    public function hasAnyActiveAd(): bool
    {
        return $this->isFeatured() || $this->isWeekendDeal() || $this->isTopUnique();
    }

    public function isAvailable($startDate, $endDate): bool
    {
        // For hotels, check room availability
        if ($this->is_hotel) {
            return $this->rooms()->active()->get()->some(function ($room) use ($startDate, $endDate) {
                return $room->hasAvailableUnits($startDate, $endDate);
            });
        }

        // Check vendor-marked unavailable dates
        if (!empty($this->unavailable_dates)) {
            $start = \Carbon\Carbon::parse($startDate);
            $end = \Carbon\Carbon::parse($endDate);

            foreach ($this->unavailable_dates as $unavailable) {
                $unavailableDate = \Carbon\Carbon::parse($unavailable);
                if ($unavailableDate->between($start, $end)) {
                    return false;
                }
            }
        }

        // Check if there are available units for the date range
        return $this->getAvailableUnits($startDate, $endDate) > 0;
    }
    
    public function getAvailableUnits($startDate, $endDate): int
    {
        // For hotels, sum available units across all rooms
        if ($this->is_hotel) {
            return $this->rooms()->active()->get()->sum(function ($room) use ($startDate, $endDate) {
                return $room->getAvailableUnits($startDate, $endDate);
            });
        }

        $start = \Carbon\Carbon::parse($startDate);
        $end = \Carbon\Carbon::parse($endDate);
        
        $bookedUnits = $this->bookings()
            ->whereIn('booking_status', ['confirmed', 'checked_in'])
            ->where('payment_status', 'paid')
            ->where(function ($query) use ($start, $end) {
                $query->whereBetween('start_date', [$start, $end])
                    ->orWhereBetween('end_date', [$start, $end])
                    ->orWhere(function ($q) use ($start, $end) {
                        $q->where('start_date', '<=', $start)
                          ->where('end_date', '>=', $end);
                    });
            })
            ->count();
        
        return max(0, $this->total_units - $bookedUnits);
    }
    
    public function hasAvailableUnits($startDate, $endDate): bool
    {
        return $this->getAvailableUnits($startDate, $endDate) > 0;
    }

    public function averageRating()
    {
        return $this->reviews()
            ->where('visible', true)
            ->avg('rating');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeVerified($query)
    {
        return $query->where('verified', true);
    }

    public function scopeFeatured($query)
    {
        return $query->whereNotNull('featured_until')
            ->where('featured_until', '>=', now());
    }

    public function scopeFeaturedNow($query)
    {
        return $query->whereNotNull('featured_until')
            ->where('featured_until', '>=', now());
    }

    public function scopeWeekendDealsNow($query)
    {
        return $query->whereNotNull('weekend_deals_until')
            ->where('weekend_deals_until', '>=', now());
    }

    public function scopeTopUniqueNow($query)
    {
        return $query->whereNotNull('top_unique_until')
            ->where('top_unique_until', '>=', now());
    }

    public function scopeInCity($query, $city)
    {
        return $query->where('city', 'like', "%{$city}%");
    }

    public function scopeInPriceRange($query, $min, $max)
    {
        return $query->whereBetween('price_per_night', [$min, $max]);
    }

    public function scopeHotels($query) // NEW
    {
        return $query->where('is_hotel', true);
    }

    public function scopeNonHotels($query) // NEW
    {
        return $query->where('is_hotel', false);
    }
}