<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CarReview extends Model
{
    use HasFactory;

    protected $fillable = [
        'car_id',
        'car_booking_id',
        'user_id',
        'overall_rating',
        'cleanliness_rating',
        'comfort_rating',
        'value_rating',
        'condition_rating',
        'comment',
        'photos',
        'is_approved',
        'approved_at',
    ];

    protected $casts = [
        'photos' => 'array',
        'is_approved' => 'boolean',
        'approved_at' => 'datetime',
    ];

    // Relationships
    public function car()
    {
        return $this->belongsTo(Car::class);
    }

    public function booking()
    {
        return $this->belongsTo(CarBooking::class, 'car_booking_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // Scopes
    public function scopeApproved($query)
    {
        return $query->where('is_approved', true);
    }

    public function scopePending($query)
    {
        return $query->where('is_approved', false);
    }

    // Methods
    public function approve()
    {
        $this->update([
            'is_approved' => true,
            'approved_at' => now(),
        ]);

        // Update car's average rating
        $this->car->updateRating();
    }

    public function getAverageRatingAttribute()
    {
        $ratings = array_filter([
            $this->cleanliness_rating,
            $this->comfort_rating,
            $this->value_rating,
            $this->condition_rating,
        ]);

        return count($ratings) > 0 ? array_sum($ratings) / count($ratings) : $this->overall_rating;
    }
}
