<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class CarBooking extends Model
{
    use HasFactory;

    protected $fillable = [
        'booking_reference',
        'car_id',
        'user_id',
        'vendor_id',
        'pickup_date',
        'pickup_time',
        'return_date',
        'return_time',
        'total_days',
        'rental_cost',
        'insurance_cost',
        'security_deposit',
        'extra_charges',
        'discount_amount',
        'promo_code',
        'total_amount',
        'driver_name',
        'driver_phone',
        'driver_email',
        'driver_license_number',
        'driver_license_expiry',
        'driver_age',
        'driver_experience_years',
        'driver_license_photo',
        'with_driver',
        'driver_cost',
        'airport_pickup',
        'airport_pickup_cost',
        'special_requests',
        'starting_mileage',
        'ending_mileage',
        'total_mileage',
        'allowed_mileage',
        'extra_mileage',
        'payment_status',
        'payment_method',
        'payment_reference',
        'payment_proof',
        'paid_at',
        'verified_at',
        'deposit_status',
        'deposit_refund_amount',
        'deposit_deduction_amount',
        'deposit_deduction_reason',
        'deposit_refunded_at',
        'admin_notes',
        'status',
        'cancellation_reason',
        'cancelled_at',
        'confirmed_at',
        'pickup_at',
        'returned_at',
        'pickup_condition_notes',
        'pickup_photos',
        'return_condition_notes',
        'return_photos',
        'damage_report',
        'commission_rate',
        'commission_amount',
        'vendor_payout_amount',
        'refund_payout_status',
        'refund_requested_at',
        'refund_paid_at',
        'refund_payment_method',
        'refund_payment_details',
    ];

    protected $casts = [
        'pickup_date' => 'date',
        'return_date' => 'date',
        'driver_license_expiry' => 'date',
        'pickup_photos' => 'array',
        'return_photos' => 'array',
        'with_driver' => 'boolean',
        'airport_pickup' => 'boolean',
        'rental_cost' => 'decimal:2',
        'insurance_cost' => 'decimal:2',
        'security_deposit' => 'decimal:2',
        'extra_charges' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'driver_cost' => 'decimal:2',
        'airport_pickup_cost' => 'decimal:2',
        'deposit_refund_amount' => 'decimal:2',
        'deposit_deduction_amount' => 'decimal:2',
        'commission_rate' => 'decimal:2',
        'commission_amount' => 'decimal:2',
        'vendor_payout_amount' => 'decimal:2',
        'paid_at' => 'datetime',
        'verified_at' => 'datetime',
        'deposit_refunded_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'confirmed_at' => 'datetime',
        'pickup_at' => 'datetime',
        'returned_at' => 'datetime',
        'refund_requested_at' => 'datetime',
        'refund_paid_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($booking) {
            if (empty($booking->booking_reference)) {
                $booking->booking_reference = 'CR' . strtoupper(Str::random(10));
            }
            if (empty($booking->payment_reference)) {
                $booking->payment_reference = 'PAY-CR-' . time() . '-' . strtoupper(Str::random(8));
            }
            
            // Calculate commission (excluding security deposit)
            if (empty($booking->commission_rate)) {
                $booking->commission_rate = \App\Models\Setting::get('car_rental_commission_rate', 15);
            }
            
            // Commission is calculated on rental amount only, not security deposit
            $rentalAmount = $booking->total_amount - ($booking->security_deposit ?? 0);
            $booking->commission_amount = ($rentalAmount * $booking->commission_rate) / 100;
            $booking->vendor_payout_amount = $rentalAmount - $booking->commission_amount;
        });
    }

    // Relationships
    public function car()
    {
        return $this->belongsTo(Car::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function vendor()
    {
        return $this->belongsTo(Vendor::class);
    }

    public function review()
    {
        return $this->hasOne(CarReview::class);
    }

    public function depositLogs()
    {
        return $this->hasMany(CarDepositLog::class, 'booking_id');
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    // Methods
    public function canBeCancelled()
    {
        return in_array($this->status, ['pending', 'confirmed']) && 
               $this->pickup_date->isFuture();
    }

    public function canBeReviewed()
    {
        return $this->status === 'completed' && 
               !$this->review()->exists();
    }

    public function calculateExtraCharges()
    {
        $charges = 0;

        // Extra mileage charges
        if ($this->extra_mileage > 0 && $this->car->extra_mileage_cost) {
            $charges += $this->extra_mileage * $this->car->extra_mileage_cost;
        }

        return $charges;
    }

    public function markAsPaid($paymentMethod, $paymentReference = null)
    {
        $this->update([
            'payment_status' => 'paid',
            'payment_method' => $paymentMethod,
            'payment_reference' => $paymentReference,
            'paid_at' => now(),
        ]);
    }

    public function confirm()
    {
        $this->update([
            'status' => 'confirmed',
            'confirmed_at' => now(),
        ]);
    }

    public function startRental($startingMileage, $photos = [])
    {
        $this->update([
            'status' => 'active',
            'pickup_at' => now(),
            'starting_mileage' => $startingMileage,
            'pickup_photos' => $photos,
        ]);
    }

    public function completeRental($endingMileage, $photos = [], $notes = null)
    {
        $totalMileage = $endingMileage - $this->starting_mileage;
        $allowedMileage = $this->total_days * ($this->car->mileage_limit_per_day ?? 0);
        $extraMileage = max(0, $totalMileage - $allowedMileage);

        $this->update([
            'status' => 'completed',
            'returned_at' => now(),
            'ending_mileage' => $endingMileage,
            'total_mileage' => $totalMileage,
            'allowed_mileage' => $allowedMileage,
            'extra_mileage' => $extraMileage,
            'return_photos' => $photos,
            'return_condition_notes' => $notes,
        ]);

        // Calculate and add extra charges
        $extraCharges = $this->calculateExtraCharges();
        if ($extraCharges > 0) {
            $this->update(['extra_charges' => $extraCharges]);
        }
    }

    public function refundDeposit($amount = null, $reason = null)
    {
        $refundAmount = $amount ?? $this->security_deposit;
        
        $this->update([
            'deposit_status' => $amount < $this->security_deposit ? 'partially_deducted' : 'refunded',
            'deposit_refund_amount' => $refundAmount,
            'deposit_deduction_amount' => $this->security_deposit - $refundAmount,
            'deposit_deduction_reason' => $reason,
            'deposit_refunded_at' => now(),
        ]);
    }

    /**
     * Check if user can request refund payout
     */
    public function canRequestRefund()
    {
        return in_array($this->deposit_status, ['refunded', 'deducted', 'partially_deducted']) &&
               $this->refund_payout_status === 'pending' &&
               $this->deposit_refund_amount > 0;
    }

    /**
     * Request refund payout
     */
    public function requestRefundPayout()
    {
        if (!$this->canRequestRefund()) {
            return false;
        }

        $this->update([
            'refund_payout_status' => 'requested',
            'refund_requested_at' => now(),
        ]);

        return true;
    }

    /**
     * Mark refund as processing
     */
    public function markRefundProcessing()
    {
        $this->update([
            'refund_payout_status' => 'processing',
        ]);
    }

    /**
     * Mark refund as paid
     */
    public function markRefundPaid($paymentMethod = null, $paymentDetails = null)
    {
        $this->update([
            'refund_payout_status' => 'paid',
            'refund_paid_at' => now(),
            'refund_payment_method' => $paymentMethod,
            'refund_payment_details' => $paymentDetails,
        ]);
    }
}