<?php

namespace App\Models;

// app/Models/Booking.php

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Booking extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'property_id',
        'vendor_id',
        'start_date',
        'end_date',
        'nights_count',
        'adults_count',
        'children_count',
        'guest_count',
        'trip_type',
        'requires_vendor_approval',
        'vendor_approval_status',
        'vendor_approved_at',
        'vendor_rejection_reason',
        'property_price_per_night',
        'subtotal',
        'caution_fee_amount',
        'total_amount',
        'payment_status',
        'payment_reference',
        'payment_method',
        'paid_at',
        'admin_verified',
        'verified_by',
        'admin_verified_at',
        'caution_refund_status',
        'caution_refund_decided_at',
        'caution_decided_by',
        'caution_refund_reason',
        'booking_status',
        'cancelled_at',
        'cancellation_reason',
        'user_visible_vendor_info',
        'vendor_visible_user_info',
        'map_visible',
        'commission_rate',
        'commission_amount',
        'vendor_payout_amount',
        'payment_proof',
        'payment_notes',
        'verified_at',
        'discount_deal_id',
        'discount_code',
        'discount_amount',
        'checked_in_at',
        'checked_out_at',
        'caution_deduction_amount',
        'caution_refund_amount',
        'caution_refunded_at',
        'caution_deduction_reason',
        'caution_deduction_photos',
        'refund_payout_status',
        'refund_requested_at',
        'refund_paid_at',
        'refund_payment_method',
        'refund_payment_details',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'property_price_per_night' => 'decimal:2',
        'subtotal' => 'decimal:2',
        'caution_fee_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'commission_rate' => 'decimal:2',
        'commission_amount' => 'decimal:2',
        'vendor_payout_amount' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'vendor_approved_at' => 'datetime',
        'paid_at' => 'datetime',
        'admin_verified_at' => 'datetime',
        'caution_refund_decided_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'requires_vendor_approval' => 'boolean',
        'admin_verified' => 'boolean',
        'user_visible_vendor_info' => 'boolean',
        'vendor_visible_user_info' => 'boolean',
        'map_visible' => 'boolean',
        'verified_at' => 'datetime',
        'checked_in_at' => 'datetime',
        'checked_out_at' => 'datetime',
        'caution_refunded_at' => 'datetime',
        'refund_requested_at' => 'datetime',
        'refund_paid_at' => 'datetime',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function property()
    {
        return $this->belongsTo(Property::class);
    }

    public function vendor()
    {
        return $this->belongsTo(Vendor::class);
    }

    public function verifiedBy()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function cautionDecidedBy()
    {
        return $this->belongsTo(User::class, 'caution_decided_by');
    }

    public function review()
    {
        return $this->hasOne(Review::class);
    }

    public function cautionRefundLogs()
    {
        return $this->hasMany(CautionRefundLog::class);
    }

    public function transaction()
    {
        return $this->hasOne(Transaction::class);
    }

    public function activities()
    {
        return $this->morphMany(Activity::class, 'subject');
    }

    public function canUnlockDetails(): bool
    {
        return $this->payment_status === 'paid' && $this->admin_verified;
    }

    public function isCompleted(): bool
    {
        return $this->booking_status === 'completed' && $this->end_date->isPast();
    }

    public function canReview(): bool
    {
        return $this->isCompleted() && is_null($this->review);
    }

    public function scopePending($query)
    {
        return $query->where('booking_status', 'pending');
    }

    public function scopeConfirmed($query)
    {
        return $query->where('booking_status', 'confirmed');
    }

    public function scopeRequiresApproval($query)
    {
        return $query->where('requires_vendor_approval', true)
            ->where('vendor_approval_status', 'pending');
    }

    public function scopePendingAdminVerification($query)
    {
        return $query->where('payment_status', 'paid')
            ->where('admin_verified', false);
    }
}