<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\CarBooking;
use App\Models\Payout;
use Illuminate\Http\Request;

class TransactionController extends Controller
{
    public function index(Request $request)
    {
        $vendor = auth()->user()->vendor;

        // Get filter parameters
        $type = $request->get('type', 'all'); // all, earnings, payouts
        $source = $request->get('source', 'all'); // all, property, car, deposit
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        // Collect all transactions
        $transactions = collect();

        // Property Bookings (Earnings)
        if (in_array($type, ['all', 'earnings']) && in_array($source, ['all', 'property'])) {
            $propertyBookings = Booking::where('vendor_id', $vendor->id)
                ->where('payment_status', 'paid')
                ->when($startDate, fn($q) => $q->whereDate('paid_at', '>=', $startDate))
                ->when($endDate, fn($q) => $q->whereDate('paid_at', '<=', $endDate))
                ->with('property')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'earning',
                        'source' => 'property',
                        'description' => 'Property Booking - ' . $booking->property->name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->vendor_payout_amount,
                        'date' => $booking->paid_at,
                        'status' => 'completed',
                        'details' => [
                            'booking_id' => $booking->id,
                            'check_in' => $booking->check_in_date,
                            'check_out' => $booking->check_out_date,
                        ]
                    ];
                });
            $transactions = $transactions->merge($propertyBookings);
        }

        // Car Bookings (Earnings)
        if (in_array($type, ['all', 'earnings']) && in_array($source, ['all', 'car'])) {
            $carBookings = CarBooking::where('vendor_id', $vendor->id)
                ->where('payment_status', 'paid')
                ->when($startDate, fn($q) => $q->whereDate('paid_at', '>=', $startDate))
                ->when($endDate, fn($q) => $q->whereDate('paid_at', '<=', $endDate))
                ->with('car')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'earning',
                        'source' => 'car',
                        'description' => 'Car Rental - ' . $booking->car->full_name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->vendor_payout_amount,
                        'date' => $booking->paid_at,
                        'status' => 'completed',
                        'details' => [
                            'booking_id' => $booking->id,
                            'pickup_date' => $booking->pickup_date,
                            'return_date' => $booking->return_date,
                        ]
                    ];
                });
            $transactions = $transactions->merge($carBookings);
        }

        // Security Deposit Deductions (Car - Earnings)
        if (in_array($type, ['all', 'earnings']) && in_array($source, ['all', 'deposit'])) {
            $depositDeductions = CarBooking::where('vendor_id', $vendor->id)
                ->whereIn('deposit_status', ['deducted', 'partially_deducted'])
                ->whereNotNull('deposit_deduction_amount')
                ->where('deposit_deduction_amount', '>', 0)
                ->when($startDate, fn($q) => $q->whereDate('deposit_refunded_at', '>=', $startDate))
                ->when($endDate, fn($q) => $q->whereDate('deposit_refunded_at', '<=', $endDate))
                ->with('car')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'earning',
                        'source' => 'deposit',
                        'description' => 'Security Deposit Deduction - ' . $booking->car->full_name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->deposit_deduction_amount,
                        'date' => $booking->deposit_refunded_at,
                        'status' => 'completed',
                        'details' => [
                            'booking_id' => $booking->id,
                            'reason' => $booking->deposit_deduction_reason,
                        ]
                    ];
                });
            $transactions = $transactions->merge($depositDeductions);
        }

        // Caution Fee Deductions (Property - Earnings)
        if (in_array($type, ['all', 'earnings']) && in_array($source, ['all', 'deposit'])) {
            $cautionDeductions = Booking::where('vendor_id', $vendor->id)
                ->whereIn('caution_refund_status', ['deducted', 'partially_deducted'])
                ->whereNotNull('caution_deduction_amount')
                ->where('caution_deduction_amount', '>', 0)
                ->when($startDate, fn($q) => $q->whereDate('caution_refunded_at', '>=', $startDate))
                ->when($endDate, fn($q) => $q->whereDate('caution_refunded_at', '<=', $endDate))
                ->with('property')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'earning',
                        'source' => 'caution',
                        'description' => 'Caution Fee Deduction - ' . $booking->property->title,
                        'reference' => $booking->id,
                        'amount' => $booking->caution_deduction_amount,
                        'date' => $booking->caution_refunded_at,
                        'status' => 'completed',
                        'details' => [
                            'booking_id' => $booking->id,
                            'reason' => $booking->caution_deduction_reason,
                        ]
                    ];
                });
            $transactions = $transactions->merge($cautionDeductions);
        }

        // Payouts
        if (in_array($type, ['all', 'payouts'])) {
            $payouts = Payout::where('vendor_id', $vendor->id)
                ->when($startDate, fn($q) => $q->whereDate('created_at', '>=', $startDate))
                ->when($endDate, fn($q) => $q->whereDate('created_at', '<=', $endDate))
                ->get()
                ->map(function ($payout) {
                    return [
                        'id' => $payout->id,
                        'type' => 'payout',
                        'source' => 'withdrawal',
                        'description' => 'Payout to ' . $payout->bank_name,
                        'reference' => $payout->reference,
                        'amount' => $payout->amount,
                        'date' => $payout->created_at,
                        'status' => $payout->status,
                        'details' => [
                            'bank' => $payout->bank_name,
                            'account' => $payout->account_number,
                        ]
                    ];
                });
            $transactions = $transactions->merge($payouts);
        }

        // Sort by date (newest first)
        $transactions = $transactions->sortByDesc('date')->values();

        // Paginate manually
        $perPage = 20;
        $currentPage = $request->get('page', 1);
        $total = $transactions->count();
        $transactions = $transactions->forPage($currentPage, $perPage);

        // Calculate summary
        $summary = [
            'total_earnings' => $vendor->total_earned,
            'total_payouts' => $vendor->total_payout,
            'pending_balance' => $vendor->pending_balance,
            'period_earnings' => $transactions->where('type', 'earning')->sum('amount'),
            'period_payouts' => $transactions->where('type', 'payout')->sum('amount'),
        ];

        return view('vendor.transactions.index', compact('transactions', 'summary', 'total', 'perPage', 'currentPage', 'type', 'source', 'startDate', 'endDate'));
    }
}
