<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\PropertyRoom;
use Illuminate\Http\Request;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class RoomController extends Controller
{
    use AuthorizesRequests;

    public function create(Property $property)
    {
        $this->authorize('update', $property);

        if (!$property->is_hotel) {
            return redirect()->route('vendor.properties.edit', $property)
                ->with('error', 'This property is not a hotel.');
        }

        return view('vendor.properties.rooms.create', compact('property'));
    }

    public function store(Request $request, Property $property)
    {
        $this->authorize('update', $property);

        $validated = $request->validate([
            'room_type' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price_per_night' => 'required|numeric|min:0',
            'max_guests' => 'required|integer|min:1',
            'total_units' => 'required|integer|min:1',
            'bedrooms' => 'required|integer|min:0',
            'beds' => 'required|integer|min:0',
            'bathrooms' => 'required|integer|min:0',
            'room_size_sqm' => 'nullable|numeric|min:0',
            'amenities' => 'nullable|array',
            'photos' => 'nullable|array',
            'photos.*' => 'image|max:5120',
        ]);

        // Handle photo uploads
        if ($request->hasFile('photos')) {
            $photoPaths = [];
            $watermarkService = new \App\Services\ImageWatermarkService();

            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('properties/rooms', 'public');
                $watermarkedPath = $watermarkService->addWatermark($path, 'center', 50);
                $photoPaths[] = $watermarkedPath;
            }
            $validated['photos'] = $photoPaths;
        }

        $property->rooms()->create($validated);

        return redirect()->route('vendor.properties.edit', $property)
            ->with('success', 'Room type added successfully!');
    }

    public function edit(Property $property, PropertyRoom $room)
    {
        $this->authorize('update', $property);

        if ($room->property_id !== $property->id) {
            abort(404);
        }

        return view('vendor.properties.rooms.edit', compact('property', 'room'));
    }

    public function update(Request $request, Property $property, PropertyRoom $room)
    {
        $this->authorize('update', $property);

        if ($room->property_id !== $property->id) {
            abort(404);
        }

        $validated = $request->validate([
            'room_type' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price_per_night' => 'required|numeric|min:0',
            'max_guests' => 'required|integer|min:1',
            'total_units' => 'required|integer|min:1',
            'bedrooms' => 'required|integer|min:0',
            'beds' => 'required|integer|min:0',
            'bathrooms' => 'required|integer|min:0',
            'room_size_sqm' => 'nullable|numeric|min:0',
            'amenities' => 'nullable|array',
            'status' => 'required|in:active,inactive',
            'photos' => 'nullable|array',
            'photos.*' => 'image|max:5120',
            'delete_photos' => 'nullable|array',
        ]);

        // Handle photo deletions
        if ($request->has('delete_photos')) {
            $existingPhotos = $room->photos ?? [];
            $photosToDelete = $request->input('delete_photos');

            foreach ($photosToDelete as $photoPath) {
                if (in_array($photoPath, $existingPhotos)) {
                    \Storage::disk('public')->delete($photoPath);
                }
            }

            $validated['photos'] = array_values(array_diff($existingPhotos, $photosToDelete));
        } else {
            $validated['photos'] = $room->photos ?? [];
        }

        // Handle new photo uploads
        if ($request->hasFile('photos')) {
            $watermarkService = new \App\Services\ImageWatermarkService();

            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('properties/rooms', 'public');
                $watermarkedPath = $watermarkService->addWatermark($path, 'center', 50);
                $validated['photos'][] = $watermarkedPath;
            }
        }

        $room->update($validated);

        return redirect()->route('vendor.properties.edit', $property)
            ->with('success', 'Room type updated successfully!');
    }

    public function destroy(Property $property, PropertyRoom $room)
    {
        $this->authorize('update', $property);

        if ($room->property_id !== $property->id) {
            abort(404);
        }

        // Check for active bookings
        $activeBookings = $room->bookings()
            ->whereIn('booking_status', ['confirmed', 'pending'])
            ->where('payment_status', 'paid')
            ->where('end_date', '>=', now())
            ->count();

        if ($activeBookings > 0) {
            return back()->with('error', 'Cannot delete room type with active bookings.');
        }

        // Delete photos
        if ($room->photos && is_array($room->photos)) {
            foreach ($room->photos as $photo) {
                \Storage::disk('public')->delete($photo);
            }
        }

        $room->delete();

        return redirect()->route('vendor.properties.edit', $property)
            ->with('success', 'Room type deleted successfully!');
    }
}