<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\Payout;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class PayoutController extends Controller
{
    public function index()
    {
        $vendor = auth()->user()->vendor;
        
        $payouts = $vendor->payouts()
            ->latest()
            ->paginate(20);

        return view('vendor.payouts.index', compact('payouts'));
    }

    public function request(Request $request)
    {
        $vendor = auth()->user()->vendor;

        $validated = $request->validate([
            'amount' => [
                'required',
                'numeric',
                'min:1000',
                'max:' . $vendor->pending_balance,
            ],
            'transaction_pin' => 'required|digits:4',
        ]);

        // Check if vendor has set transaction PIN
        if (!$vendor->transaction_pin) {
            return back()->withErrors([
                'transaction_pin' => 'Please set your transaction PIN in your profile before requesting payout.'
            ])->withInput();
        }

        // Verify transaction PIN
        if (!Hash::check($validated['transaction_pin'], $vendor->transaction_pin)) {
            return back()->withErrors([
                'transaction_pin' => 'Invalid transaction PIN. Please try again.'
            ])->withInput();
        }

        // Check if vendor has bank details
        if (!$vendor->bank_name || !$vendor->account_number) {
            return back()->withErrors([
                'bank_details' => 'Please update your bank details before requesting payout.'
            ])->withInput();
        }

        // Create payout request
        Payout::create([
            'vendor_id' => $vendor->id,
            'amount' => $validated['amount'],
            'status' => 'pending',
            'bank_name' => $vendor->bank_name,
            'account_number' => $vendor->account_number,
            'account_name' => $vendor->account_name,
            'reference' => 'PO-' . Str::upper(Str::random(12)),
        ]);

        // Deduct from pending balance
        $vendor->decrement('pending_balance', $validated['amount']);

        return back()->with('success', 'Payout request submitted successfully. It will be reviewed by admin.');
    }
}