<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class KycController extends Controller
{
    use AuthorizesRequests;
    
    public function show()
    {
        $vendor = auth()->user()->vendor;
        $formFields = \App\Models\KycFormField::where('is_active', true)
            ->orderBy('order')
            ->get();
        
        return view('vendor.kyc.show', compact('vendor', 'formFields'));
    }
    
    public function submit(Request $request)
    {
        $vendor = auth()->user()->vendor;
        
        $rules = [
            'id_card_type' => 'required|in:NIN,Drivers License,International Passport,Voters Card',
            'id_card_number' => 'required|string|max:50',
            'id_card_document' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120',
        ];
        
        // Add property KYC validation if vendor offers properties
        if ($vendor->offersProperties()) {
            $rules['cac_document'] = 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120';
            $rules['business_address'] = 'required|string|max:500';
            $rules['home_address'] = 'required|string|max:500';
        }
        
        // Add car rental KYC validation if vendor offers car rentals
        if ($vendor->offersCarRentals()) {
            $rules['car_rental_business_name'] = 'required|string|max:255';
            $rules['car_rental_business_registration'] = 'required|string|max:100';
            $rules['car_rental_tax_id'] = 'required|string|max:100';
            $rules['car_rental_business_address'] = 'required|string|max:500';
            $rules['car_rental_business_certificate'] = 'required|file|mimes:pdf,jpg,jpeg,png|max:5120';
            $rules['car_rental_license'] = 'required|file|mimes:pdf,jpg,jpeg,png|max:5120';
            $rules['car_rental_insurance_certificate'] = 'required|file|mimes:pdf,jpg,jpeg,png|max:5120';
            $rules['car_rental_bank_name'] = 'required|string|max:100';
            $rules['car_rental_account_number'] = 'required|string|max:20';
            $rules['car_rental_account_name'] = 'required|string|max:255';
        }
        
        $validated = $request->validate($rules);
        
        // Upload ID card document
        if ($request->hasFile('id_card_document')) {
            $path = $request->file('id_card_document')->store('kyc/id_cards', 'public');
            $validated['id_card_document'] = $path;
        }
        
        // Upload CAC document if provided
        if ($request->hasFile('cac_document')) {
            $path = $request->file('cac_document')->store('kyc/cac', 'public');
            $validated['cac_document'] = $path;
        }
        
        // Upload car rental documents if vendor offers car rentals
        if ($vendor->offersCarRentals()) {
            if ($request->hasFile('car_rental_business_certificate')) {
                $path = $request->file('car_rental_business_certificate')->store('kyc/car_rental/certificates', 'public');
                $validated['car_rental_business_certificate'] = $path;
            }
            
            if ($request->hasFile('car_rental_license')) {
                $path = $request->file('car_rental_license')->store('kyc/car_rental/licenses', 'public');
                $validated['car_rental_license'] = $path;
            }
            
            if ($request->hasFile('car_rental_insurance_certificate')) {
                $path = $request->file('car_rental_insurance_certificate')->store('kyc/car_rental/insurance', 'public');
                $validated['car_rental_insurance_certificate'] = $path;
            }
            
            $validated['car_rental_kyc_status'] = 'pending';
        }
        
        $validated['kyc_status'] = 'pending';
        $validated['kyc_submitted_at'] = now();
        
        $vendor->update($validated);
        
        $message = 'KYC documents submitted successfully! We will review them shortly.';
        if ($vendor->offersCarRentals()) {
            $message = 'Property and Car Rental KYC documents submitted successfully! We will review them shortly.';
        }
        
        return redirect()->route('vendor.kyc.show')
            ->with('success', $message);
    }
}
