<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        $vendor = auth()->user()->vendor;
        
        // Check what services vendor offers
        $offersProperties = $vendor->offersProperties();
        $offersCarRentals = $vendor->offersCarRentals();

        $stats = [
            'total_properties' => $offersProperties ? $vendor->properties()->count() : 0,
            'active_properties' => $offersProperties ? $vendor->properties()->where('status', 'active')->count() : 0,
            'total_cars' => $offersCarRentals ? $vendor->cars()->count() : 0,
            'active_cars' => $offersCarRentals ? $vendor->cars()->where('status', 'approved')->where('is_available', true)->count() : 0,
            'total_bookings' => ($offersProperties ? $vendor->bookings()->count() : 0) + 
                               ($offersCarRentals ? $vendor->carBookings()->count() : 0),
            'pending_bookings' => ($offersProperties ? $vendor->bookings()
                ->where('requires_vendor_approval', true)
                ->where('vendor_approval_status', 'pending')
                ->count() : 0) + 
                ($offersCarRentals ? $vendor->carBookings()->where('status', 'pending')->count() : 0),
        ];

        // Get recent bookings based on service type
        if ($offersProperties && $offersCarRentals) {
            // Mixed vendor - show both
            $recentPropertyBookings = $vendor->bookings()
                ->with(['user', 'property'])
                ->latest()
                ->limit(5)
                ->get();
            
            $recentCarBookings = $vendor->carBookings()
                ->with(['user', 'car'])
                ->latest()
                ->limit(5)
                ->get();
                
            $recentBookings = $recentPropertyBookings->merge($recentCarBookings)->sortByDesc('created_at')->take(10);
        } elseif ($offersProperties) {
            // Property only
            $recentBookings = $vendor->bookings()
                ->with(['user', 'property'])
                ->latest()
                ->limit(10)
                ->get();
        } else {
            // Car rental only
            $recentBookings = $vendor->carBookings()
                ->with(['user', 'car'])
                ->latest()
                ->limit(10)
                ->get();
        }

        // Get top performing items based on service type
        $topProperties = $offersProperties ? $vendor->properties()
            ->withCount('bookings')
            ->orderBy('bookings_count', 'desc')
            ->limit(5)
            ->get() : collect();

        $topCars = $offersCarRentals ? $vendor->cars()
            ->withCount('bookings')
            ->orderBy('bookings_count', 'desc')
            ->limit(5)
            ->get() : collect();

        return view('vendor.dashboard', compact('stats', 'recentBookings', 'topProperties', 'topCars', 'offersProperties', 'offersCarRentals'));
    }
}
