<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\CarBooking;
use App\Services\BookingService;
use App\Models\CautionRefundLog;
use Illuminate\Http\Request;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class BookingController extends Controller
{
    use AuthorizesRequests;
    public function index(Request $request)
    {
        $vendor = auth()->user()->vendor;
        
        // Check what services vendor offers
        $offersProperties = $vendor->offersProperties();
        $offersCarRentals = $vendor->offersCarRentals();
        
        // Get property bookings only if vendor offers properties
        if ($offersProperties) {
            $query = $vendor->bookings()->with(['property', 'user']);
            
            // Filter by status
            if ($request->filled('status')) {
                $status = $request->status;
                if (in_array($status, ['pending', 'pending_payment', 'confirmed', 'checked_in', 'completed', 'cancelled'])) {
                    $query->where('booking_status', $status);
                }
            }
            
            $bookings = $query->latest()->paginate(20, ['*'], 'properties_page')->withQueryString();
        } else {
            $bookings = new \Illuminate\Pagination\LengthAwarePaginator([], 0, 20);
        }
        
        // Get car bookings only if vendor offers car rentals
        if ($offersCarRentals) {
            $carQuery = $vendor->carBookings()->with(['car.category', 'user']);
            
            // Filter car bookings by status
            if ($request->filled('status')) {
                $status = $request->status;
                if (in_array($status, ['pending', 'confirmed', 'active', 'completed', 'cancelled'])) {
                    $carQuery->where('status', $status);
                }
            }
            
            $carBookings = $carQuery->latest()->paginate(20, ['*'], 'cars_page')->withQueryString();
        } else {
            $carBookings = new \Illuminate\Pagination\LengthAwarePaginator([], 0, 20);
        }
        
        return view('vendor.bookings.index', compact('bookings', 'carBookings', 'offersProperties', 'offersCarRentals'));
    }
    
    public function show(Booking $booking)
    {
        $this->authorize('view', $booking);
        
        $booking->load(['property', 'user', 'cautionRefundLogs.actor']);
        
        return view('vendor.bookings.show', compact('booking'));
    }
    
    public function approve(Booking $booking, BookingService $bookingService)
    {
        $this->authorize('approve', $booking);
        
        if ($bookingService->approveBooking($booking, auth()->user()->vendor)) {
            return back()->with('success', 'Booking approved! User will receive payment link.');
        }
        
        return back()->withErrors(['error' => 'Failed to approve booking.']);
    }
    
    public function reject(Request $request, Booking $booking, BookingService $bookingService)
    {
        $this->authorize('reject', $booking);
        
        $validated = $request->validate([
            'reason' => 'required|string|max:500',
        ]);
        
        if ($bookingService->rejectBooking($booking, auth()->user()->vendor, $validated['reason'])) {
            return back()->with('success', 'Booking rejected.');
        }
        
        return back()->withErrors(['error' => 'Failed to reject booking.']);
    }
    
    public function decideCautionRefund(Request $request, Booking $booking)
    {
        $this->authorize('decideCautionRefund', $booking);
        
        $maxAmount = $booking->caution_fee ?? $booking->caution_fee_amount ?? 0;
        
        $validated = $request->validate([
            'has_damage' => 'required|boolean',
            'damage_description' => 'required_if:has_damage,1|nullable|string|max:1000',
            'deduction_amount' => 'required_if:has_damage,1|nullable|numeric|min:0|max:' . $maxAmount,
            'damage_photos.*' => 'nullable|image|max:5120', // 5MB max per image
        ]);

        $hasDamage = $validated['has_damage'];
        
        if ($hasDamage) {
            // Damage found - request deduction
            $photos = [];
            if ($request->hasFile('damage_photos')) {
                foreach ($request->file('damage_photos') as $photo) {
                    $photos[] = $photo->store('bookings/damages', 'public');
                }
            }

            $booking->update([
                'caution_refund_status' => 'pending_vendor',
                'caution_deduction_amount' => $validated['deduction_amount'],
                'caution_deduction_reason' => $validated['damage_description'],
                'caution_deduction_photos' => !empty($photos) ? json_encode($photos) : null,
            ]);

            // Log the action
            CautionRefundLog::create([
                'booking_id' => $booking->id,
                'actor_id' => auth()->id(),
                'actor_role' => 'vendor',
                'action' => 'overridden',
                'previous_status' => 'pending',
                'new_status' => 'pending',
                'amount' => $validated['deduction_amount'],
                'reason' => $validated['damage_description'],
            ]);

            return back()->with('success', 'Damage report submitted. Admin will review your claim.');
        } else {
            // No damage - approve full refund
            $cautionAmount = $booking->caution_fee_amount ?? $booking->caution_fee ?? 0;
            
            $booking->update([
                'caution_refund_status' => 'refunded',
                'caution_refund_amount' => $cautionAmount,
                'caution_refunded_at' => now(),
            ]);

            // Log the action
            CautionRefundLog::create([
                'booking_id' => $booking->id,
                'actor_id' => auth()->id(),
                'actor_role' => 'vendor',
                'action' => 'refunded',
                'previous_status' => 'pending',
                'new_status' => 'refunded',
                'amount' => $cautionAmount,
                'reason' => 'No damage found - full refund approved',
            ]);

            return back()->with('success', 'Caution fee approved for full refund to guest.');
        }
    }

    public function showCarBooking(CarBooking $booking)
    {
        // Check if vendor owns this booking
        if ($booking->vendor_id !== auth()->user()->vendor->id) {
            abort(403);
        }

        $booking->load(['car.category', 'user']);

        return view('vendor.car-bookings.show', compact('booking'));
    }

    public function reportDamage(Request $request, CarBooking $booking)
    {
        // Check if vendor owns this booking
        if ($booking->vendor_id !== auth()->user()->vendor->id) {
            abort(403);
        }

        // Verify booking is in pending_inspection status
        if ($booking->status !== 'pending_inspection') {
            return back()->withErrors(['error' => 'Booking must be in pending inspection status.']);
        }

        $validated = $request->validate([
            'has_damage' => 'required|boolean',
            'damage_report' => 'required_if:has_damage,1|nullable|string|max:2000',
            'deposit_deduction_amount' => 'required_if:has_damage,1|nullable|numeric|min:0|max:' . $booking->security_deposit,
            'damage_photos.*' => 'nullable|image|max:5120',
        ]);

        $damagePhotos = [];
        if ($request->hasFile('damage_photos')) {
            foreach ($request->file('damage_photos') as $photo) {
                $damagePhotos[] = $photo->store('car-bookings/damage-photos', 'public');
            }
        }

        if ($validated['has_damage']) {
            // Damage found - request admin approval for deposit deduction
            $booking->update([
                'status' => 'pending_deposit_resolution',
                'deposit_status' => 'dispute_pending',
                'damage_report' => $validated['damage_report'],
                'deposit_deduction_amount' => $validated['deposit_deduction_amount'],
                'deposit_deduction_reason' => $validated['damage_report'],
                'return_photos' => $damagePhotos,
            ]);

            return redirect()->route('vendor.car-bookings.show', $booking)
                ->with('success', 'Damage report submitted. Admin will review and notify the customer for confirmation.');
        } else {
            // No damage - process refund
            $booking->update([
                'status' => 'completed',
                'deposit_status' => 'refunded',
                'deposit_refund_amount' => $booking->security_deposit,
                'deposit_refunded_at' => now(),
            ]);

            return redirect()->route('vendor.car-bookings.show', $booking)
                ->with('success', 'Vehicle inspection completed. Security deposit will be refunded to the customer.');
        }
    }
}