<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\VendorAd;
use App\Models\AdPlan;
use App\Services\Payment\PaymentFactory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class AdController extends Controller
{
    public function index()
    {
        $activePlans = AdPlan::where('active', true)->orderBy('placement_type')->orderBy('price')->get();
        $hasActivePlans = $activePlans->count() > 0;
        
        return view('vendor.ads.index', compact('activePlans', 'hasActivePlans'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'property_id' => 'required|exists:properties,id',
            'ad_plan_id' => 'required|exists:ad_plans,id',
            'placement_type' => 'required|in:featured,weekend_deals,top_unique',
        ]);

        $vendor = auth()->user()->vendor;
        $adPlan = AdPlan::findOrFail($validated['ad_plan_id']);

        // Verify placement type matches the plan
        if ($adPlan->placement_type !== $validated['placement_type']) {
            return back()->withErrors(['placement_type' => 'Invalid placement type for this plan.']);
        }

        // Check if property belongs to vendor
        $property = $vendor->properties()->findOrFail($validated['property_id']);

        // Check if property already has an active ad for this placement type
        $existingAd = VendorAd::where('vendor_id', $vendor->id)
            ->where('property_id', $property->id)
            ->where('placement_type', $validated['placement_type'])
            ->where('active', true)
            ->first();

        if ($existingAd) {
            return back()->withErrors([
                'property_id' => 'This property already has an active ' . str_replace('_', ' ', $validated['placement_type']) . ' advertisement.'
            ]);
        }

        // Create ad record
        $vendorAd = VendorAd::create([
            'vendor_id' => $vendor->id,
            'property_id' => $property->id,
            'ad_plan_id' => $adPlan->id,
            'placement_type' => $validated['placement_type'],
            'payment_status' => 'pending',
            'payment_reference' => 'AD-' . Str::upper(Str::random(12)),
        ]);

        // Redirect to payment selection page
        return redirect()->route('vendor.ads.payment', $vendorAd);
    }

    public function showPayment(VendorAd $vendorAd)
    {
        // Ensure the ad belongs to the authenticated vendor
        if ($vendorAd->vendor_id !== auth()->user()->vendor->id) {
            abort(403);
        }

        // Get available payment methods
        $availablePaymentMethods = \App\Models\PaymentMethod::where('is_active', true)
            ->orderBy('display_order')
            ->get()
            ->map(function ($method) {
                return [
                    'id' => $method->slug,
                    'name' => $method->name,
                    'description' => $method->description ?? 'Pay securely with ' . $method->name,
                    'settings' => $method->settings ?? [],
                ];
            })
            ->toArray();

        return view('vendor.ads.payment', compact('vendorAd', 'availablePaymentMethods'));
    }

    public function processPayment(Request $request, VendorAd $vendorAd)
    {
        // Ensure the ad belongs to the authenticated vendor
        if ($vendorAd->vendor_id !== auth()->user()->vendor->id) {
            abort(403);
        }

        $validated = $request->validate([
            'payment_method' => 'required|string',
            'promo_code' => 'nullable|string',
        ]);

        // Calculate final amount with promo code if provided
        $originalAmount = $vendorAd->adPlan->price;
        $finalAmount = $originalAmount;
        $promoCodeId = null;
        $promoCode = null;
        $discountAmount = 0;

        if (!empty($validated['promo_code'])) {
            $promoService = app(\App\Services\PromoCodeService::class);
            $validation = $promoService->validateCode(
                $validated['promo_code'],
                auth()->user()->vendor,
                $originalAmount
            );

            if ($validation['valid']) {
                $promoCodeId = $validation['promo_code']->id;
                $promoCode = $validation['promo_code']->code;
                $discountAmount = $validation['discount_amount'];
                $finalAmount = $validation['final_amount'];
            }
        }

        // Handle bank transfer separately
        if ($validated['payment_method'] === 'bank_transfer') {
            // Update payment method and promo code info
            $vendorAd->update([
                'payment_method' => 'bank_transfer',
                'promo_code_id' => $promoCodeId,
                'promo_code' => $promoCode,
                'discount_amount' => $discountAmount,
                'final_amount' => $finalAmount,
            ]);
            
            // Redirect to bank transfer instructions page
            return redirect()->route('vendor.ads.bank-transfer', $vendorAd)
                ->with('info', 'Please complete the bank transfer and upload proof of payment.');
        }

        // Handle online payment gateways
        try {
            // Update payment method and promo code info
            $vendorAd->update([
                'payment_method' => $validated['payment_method'],
                'promo_code_id' => $promoCodeId,
                'promo_code' => $promoCode,
                'discount_amount' => $discountAmount,
                'final_amount' => $finalAmount,
            ]);
            
            $paymentService = PaymentFactory::create($validated['payment_method']);
            
            $result = $paymentService->initializePayment([
                'email' => auth()->user()->email,
                'amount' => $finalAmount, // Use discounted amount
                'reference' => $vendorAd->payment_reference,
                'callback_url' => route('payment.callback'),
                'metadata' => [
                    'vendor_ad_id' => $vendorAd->id,
                    'vendor_id' => $vendorAd->vendor_id,
                    'type' => 'ad_payment',
                    'placement_type' => $vendorAd->placement_type,
                    'promo_code' => $promoCode,
                    'discount_amount' => $discountAmount,
                    'original_amount' => $originalAmount,
                ],
            ]);

            return redirect($result['payment_url']);
        } catch (\Exception $e) {
            \Log::error('Ad payment initialization failed: ' . $e->getMessage(), [
                'vendor_ad_id' => $vendorAd->id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()->withErrors(['payment' => 'Failed to initialize payment: ' . $e->getMessage()]);
        }
    }

    public function showBankTransfer(VendorAd $vendorAd)
    {
        // Ensure the ad belongs to the authenticated vendor
        if ($vendorAd->vendor_id !== auth()->user()->vendor->id) {
            abort(403);
        }

        // Get bank transfer payment method details
        $bankTransferMethod = \App\Models\PaymentMethod::where('slug', 'bank_transfer')
            ->where('is_active', true)
            ->first();

        if (!$bankTransferMethod) {
            return redirect()->route('vendor.ads.index')
                ->with('error', 'Bank transfer is not available at the moment.');
        }

        return view('vendor.ads.bank-transfer', compact('vendorAd', 'bankTransferMethod'));
    }

    public function uploadProof(Request $request, VendorAd $vendorAd)
    {
        // Ensure the ad belongs to the authenticated vendor
        if ($vendorAd->vendor_id !== auth()->user()->vendor->id) {
            abort(403);
        }

        $validated = $request->validate([
            'payment_proof' => 'required|image|max:5120', // 5MB max
        ]);

        // Store the payment proof
        $path = $request->file('payment_proof')->store('payment-proofs', 'public');
        
        // Update with payment proof - status remains 'pending' until admin verifies
        $vendorAd->update([
            'payment_proof' => $path,
            // payment_status stays 'pending' - admin will change to 'paid' after verification
        ]);

        return redirect()->route('vendor.ads.index')
            ->with('success', 'Payment proof uploaded successfully. Your payment will be verified shortly.');
    }
}