<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\CarBooking;
use App\Models\VendorAd;
use App\Services\Payment\PaymentFactory;
use Illuminate\Http\Request;

class PaymentCallbackController extends Controller
{
    public function callback(Request $request)
    {
        $reference = $request->query('reference') ?? $request->query('tx_ref');
        
        if (!$reference) {
            return redirect()->route('home')
                ->withErrors(['error' => 'Invalid payment reference.']);
        }
        
        // Determine payment provider
        $provider = $request->query('provider', 'paystack');
        $paymentService = PaymentFactory::create($provider);
        
        try {
            $verification = $paymentService->verifyPayment($reference);
            
            if ($verification['status'] !== 'success' && $verification['status'] !== 'successful') {
                return redirect()->route('home')
                    ->withErrors(['error' => 'Payment verification failed.']);
            }
            
            // Find booking, car booking, or ad
            $booking = Booking::where('payment_reference', $reference)->first();
            
            if ($booking) {
                // Update booking status for online payments
                $booking->update([
                    'payment_status' => 'paid',
                    'paid_at' => now(),
                    'booking_status' => 'confirmed',
                    'admin_verified' => true,
                    'user_visible_vendor_info' => true,
                    'vendor_visible_user_info' => true,
                    'map_visible' => true,
                ]);
                
                // Credit vendor balance
                $vendor = $booking->vendor;
                $vendor->increment('pending_balance', $booking->vendor_payout_amount);
                $vendor->increment('total_earned', $booking->vendor_payout_amount);
                
                return redirect()->route('bookings.show', $booking)
                    ->with('success', 'Payment successful! Your booking is confirmed.');
            }
            
            // Check for car booking
            $carBooking = CarBooking::where('payment_reference', $reference)->first();
            
            if ($carBooking) {
                // Update car booking status for online payments
                $carBooking->update([
                    'payment_status' => 'paid',
                    'paid_at' => now(),
                    'status' => 'confirmed',
                    'confirmed_at' => now(),
                ]);
                
                // Credit vendor balance (after commission)
                $vendor = $carBooking->vendor;
                if ($vendor) {
                    $vendor->increment('pending_balance', $carBooking->vendor_payout_amount);
                    $vendor->increment('total_earned', $carBooking->vendor_payout_amount);
                }
                
                return redirect()->route('cars.booking.show', $carBooking)
                    ->with('success', 'Payment successful! Your car booking is confirmed.');
            }
            
            $vendorAd = VendorAd::where('payment_reference', $reference)->first();
            
            if ($vendorAd) {
                // Update ad payment status - online payments are automatically verified
                $vendorAd->update([
                    'payment_status' => 'paid',
                    'paid_at' => now(),
                ]);
                
                return redirect()->route('vendor.ads.index')
                    ->with('success', 'Payment successful! Your ad is pending admin approval.');
            }
            
            return redirect()->route('home')
                ->with('info', 'Payment processed successfully.');
                
        } catch (\Exception $e) {
            return redirect()->route('home')
                ->withErrors(['error' => 'Payment verification failed.']);
        }
    }
}