<?php

namespace App\Http\Controllers;

use App\Models\CarBooking;
use App\Models\CarReview;
use Illuminate\Http\Request;

class CarReviewController extends Controller
{
    public function create(CarBooking $booking)
    {
        // Check if user owns this booking
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Check if booking can be reviewed
        if (!$booking->canBeReviewed()) {
            return redirect()->route('bookings.index')
                ->withErrors(['error' => 'This booking cannot be reviewed yet.']);
        }

        // Check if already reviewed
        if ($booking->review) {
            return redirect()->route('bookings.index')
                ->withErrors(['error' => 'You have already reviewed this booking.']);
        }

        $booking->load(['car', 'car.vendor']);

        return view('cars.reviews.create', compact('booking'));
    }

    public function store(Request $request, CarBooking $booking)
    {
        // Check if user owns this booking
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Check if booking can be reviewed
        if (!$booking->canBeReviewed()) {
            return redirect()->route('bookings.index')
                ->withErrors(['error' => 'This booking cannot be reviewed yet.']);
        }

        // Check if already reviewed
        if ($booking->review) {
            return redirect()->route('bookings.index')
                ->withErrors(['error' => 'You have already reviewed this booking.']);
        }

        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'required|string|min:10|max:1000',
        ]);

        CarReview::create([
            'car_booking_id' => $booking->id,
            'car_id' => $booking->car_id,
            'user_id' => auth()->id(),
            'overall_rating' => $validated['rating'],
            'comment' => $validated['comment'],
            'is_approved' => false, // Requires admin approval
        ]);

        return redirect()->route('bookings.index')
            ->with('success', 'Thank you for your review! It will be published after admin approval.');
    }
}
