<?php

namespace App\Http\Controllers;

use App\Models\Car;
use App\Models\CarCategory;
use App\Models\CarBooking;
use App\Models\Setting;
use Illuminate\Http\Request;

class CarRentalController extends Controller
{
    public function index(Request $request)
    {
        // Check if car rental is enabled
        if (!Setting::get('car_rental_enabled', false)) {
            abort(404);
        }

        $query = Car::with(['category', 'vendor'])
            ->where('status', 'approved')
            ->where('is_available', true);
        
        // Filter by date availability if dates are provided
        if ($request->filled('pickup_date') && $request->filled('return_date')) {
            $pickupDate = $request->pickup_date;
            $returnDate = $request->return_date;
            
            // Exclude cars with conflicting bookings
            $query->whereDoesntHave('bookings', function($q) use ($pickupDate, $returnDate) {
                $q->where('payment_status', 'paid')
                  ->whereIn('status', ['pending', 'confirmed', 'active'])
                  ->whereNull('returned_at')
                  ->where(function ($query) use ($pickupDate, $returnDate) {
                      $query->whereBetween('pickup_date', [$pickupDate, $returnDate])
                          ->orWhereBetween('return_date', [$pickupDate, $returnDate])
                          ->orWhere(function ($q) use ($pickupDate, $returnDate) {
                              $q->where('pickup_date', '<=', $pickupDate)
                                ->where('return_date', '>=', $returnDate);
                          });
                  });
            });
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('car_category_id', $request->category);
        }

        // Filter by location
        if ($request->filled('state')) {
            $query->where('state', 'like', '%' . $request->state . '%');
        }

        if ($request->filled('city')) {
            $query->where('city', 'like', '%' . $request->city . '%');
        }

        // Filter by price range
        if ($request->filled('min_price')) {
            $query->where('price_per_day', '>=', $request->min_price);
        }

        if ($request->filled('max_price')) {
            $query->where('price_per_day', '<=', $request->max_price);
        }

        // Filter by transmission
        if ($request->filled('transmission')) {
            $query->where('transmission', $request->transmission);
        }

        // Filter by fuel type
        if ($request->filled('fuel_type')) {
            $query->where('fuel_type', $request->fuel_type);
        }

        // Filter by seats
        if ($request->filled('seats')) {
            $query->where('seats', '>=', $request->seats);
        }

        // Sort
        $sortBy = $request->get('sort', 'latest');
        switch ($sortBy) {
            case 'price_low':
                $query->orderBy('price_per_day', 'asc');
                break;
            case 'price_high':
                $query->orderBy('price_per_day', 'desc');
                break;
            case 'popular':
                $query->orderBy('total_bookings', 'desc');
                break;
            case 'rating':
                $query->orderBy('average_rating', 'desc');
                break;
            default:
                $query->latest();
        }

        $cars = $query->paginate(12)->withQueryString();
        $categories = CarCategory::active()->ordered()->get();

        // Get unique states for filter
        $states = Car::where('status', 'approved')
            ->distinct()
            ->pluck('state')
            ->sort()
            ->values();

        return view('cars.index', compact('cars', 'categories', 'states'));
    }

    public function show(Car $car)
    {
        // Check if car rental is enabled
        if (!Setting::get('car_rental_enabled', false)) {
            abort(404);
        }

        // Check if car is approved and available
        if ($car->status !== 'approved') {
            abort(404);
        }

        // Load relationships
        $car->load(['category', 'vendor', 'approvedReviews.user']);

        // Increment views
        $car->incrementViews();

        // Get similar cars
        $similarCars = Car::where('status', 'approved')
            ->where('is_available', true)
            ->where('id', '!=', $car->id)
            ->where('car_category_id', $car->car_category_id)
            ->where('state', $car->state)
            ->limit(4)
            ->get();

        return view('cars.show', compact('car', 'similarCars'));
    }

    public function booking(Car $car)
    {
        // Check if car rental is enabled
        if (!Setting::get('car_rental_enabled', false)) {
            abort(404);
        }

        // Check if car is approved and available
        if ($car->status !== 'approved' || !$car->is_available) {
            return redirect()->route('cars.show', $car)
                ->with('error', 'This car is not available for booking.');
        }

        // Check if user is authenticated
        if (!auth()->check()) {
            return redirect()->route('login')
                ->with('info', 'Please login to book a car.');
        }

        $car->load(['category', 'vendor']);

        return view('cars.booking', compact('car'));
    }

    public function storeBooking(Request $request, Car $car)
    {
        // Check if car rental is enabled
        if (!Setting::get('car_rental_enabled', false)) {
            abort(404);
        }

        // Check if car is available
        if ($car->status !== 'approved' || !$car->is_available) {
            return back()->with('error', 'This car is not available for booking.');
        }

        // Validate request
        $validated = $request->validate([
            'pickup_date' => 'required|date|after:today',
            'pickup_time' => 'required|date_format:H:i',
            'return_date' => 'required|date|after:pickup_date',
            'return_time' => 'required|date_format:H:i',
            'driver_name' => 'required|string|max:255',
            'driver_phone' => 'required|string|max:20',
            'driver_email' => 'required|email|max:255',
            'driver_license_number' => 'required|string|max:50',
            'driver_license_expiry' => 'required|date|after:return_date',
            'driver_age' => 'required|integer|min:' . $car->minimum_driver_age,
            'driver_experience_years' => 'required|integer|min:' . $car->minimum_license_years,
            'driver_license_photo' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'with_driver' => 'boolean',
            'airport_pickup' => 'boolean',
            'special_requests' => 'nullable|string|max:1000',
            'include_insurance' => 'boolean',
        ]);

        // Check availability for dates
        $pickupDateTime = $validated['pickup_date'] . ' ' . $validated['pickup_time'];
        $returnDateTime = $validated['return_date'] . ' ' . $validated['return_time'];

        if (!$car->isAvailableForDates($pickupDateTime, $returnDateTime)) {
            return back()->with('error', 'Car is not available for the selected dates.');
        }

        // Calculate costs
        $totalDays = ceil((strtotime($returnDateTime) - strtotime($pickupDateTime)) / 86400);
        
        // Check minimum rental days
        if ($totalDays < $car->minimum_rental_days) {
            return back()->withErrors(['return_date' => 'Minimum rental period is ' . $car->minimum_rental_days . ' days. You selected ' . $totalDays . ' days.'])->withInput();
        }

        // Check maximum rental days
        if ($car->maximum_rental_days && $car->maximum_rental_days > 0 && $totalDays > $car->maximum_rental_days) {
            return back()->withErrors(['return_date' => 'Maximum rental period is ' . $car->maximum_rental_days . ' days. You selected ' . $totalDays . ' days. Please contact the vendor for longer rentals.'])->withInput();
        }

        $costs = $car->calculateRentalCost($totalDays);
        
        $rentalCost = $costs['rental_cost'];
        $insuranceCost = $request->has('include_insurance') ? $costs['insurance_cost'] : 0;
        $securityDeposit = $car->security_deposit;
        $totalAmount = $rentalCost + $insuranceCost + $securityDeposit;

        // Upload driver license photo
        $licensePath = $request->file('driver_license_photo')->store('bookings/licenses', 'public');

        // Calculate commission and vendor payout
        $commissionRate = $car->vendor->commission_rate ?? 10; // Default 10%
        $commissionAmount = ($rentalCost * $commissionRate) / 100;
        $vendorPayoutAmount = $rentalCost - $commissionAmount;

        // Create booking
        $booking = CarBooking::create([
            'car_id' => $car->id,
            'user_id' => auth()->id(),
            'vendor_id' => $car->vendor_id,
            'pickup_date' => $validated['pickup_date'],
            'pickup_time' => $validated['pickup_time'],
            'return_date' => $validated['return_date'],
            'return_time' => $validated['return_time'],
            'total_days' => $totalDays,
            'rental_cost' => $rentalCost,
            'insurance_cost' => $insuranceCost,
            'security_deposit' => $securityDeposit,
            'total_amount' => $totalAmount,
            'commission_rate' => $commissionRate,
            'commission_amount' => $commissionAmount,
            'vendor_payout_amount' => $vendorPayoutAmount,
            'driver_name' => $validated['driver_name'],
            'driver_phone' => $validated['driver_phone'],
            'driver_email' => $validated['driver_email'],
            'driver_license_number' => $validated['driver_license_number'],
            'driver_license_expiry' => $validated['driver_license_expiry'],
            'driver_age' => $validated['driver_age'],
            'driver_experience_years' => $validated['driver_experience_years'],
            'driver_license_photo' => $licensePath,
            'with_driver' => $request->has('with_driver'),
            'airport_pickup' => $request->has('airport_pickup'),
            'special_requests' => $validated['special_requests'] ?? null,
            'status' => 'pending',
            'payment_status' => 'pending',
        ]);

        // Redirect to payment
        return redirect()->route('cars.booking.payment', $booking)
            ->with('success', 'Booking created successfully! Please proceed with payment.');
    }

    public function payment(CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Check if already paid
        if ($booking->payment_status === 'paid') {
            return redirect()->route('user.bookings.show', $booking)
                ->with('info', 'This booking has already been paid.');
        }

        $booking->load(['car.category', 'car.vendor']);

        return view('cars.payment', compact('booking'));
    }

    public function processPayment(Request $request, CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Check if already paid
        if ($booking->payment_status === 'paid') {
            return redirect()->route('user.bookings.show', $booking)
                ->with('info', 'This booking has already been paid.');
        }

        // Get active payment method slugs for validation
        $activePaymentMethods = \App\Models\PaymentMethod::active()->pluck('slug')->toArray();
        
        $validated = $request->validate([
            'payment_method' => 'required|in:' . implode(',', $activePaymentMethods),
        ]);
        
        // Update booking with selected payment method
        $booking->update(['payment_method' => $validated['payment_method']]);
        
        // Handle bank transfer differently - no gateway needed
        if ($validated['payment_method'] === 'bank_transfer') {
            // Update booking status to awaiting payment confirmation
            $booking->update([
                'payment_status' => 'pending_verification',
                'status' => 'pending',
            ]);
            
            return redirect()->route('cars.booking.bank-transfer', $booking)
                ->with('success', 'Please complete the bank transfer and upload proof of payment.');
        }
        
        // Initialize payment for online gateways (Paystack, Flutterwave)
        $paymentService = \App\Services\Payment\PaymentFactory::create($validated['payment_method']);
        
        try {
            $result = $paymentService->initializePayment([
                'email' => auth()->user()->email,
                'amount' => $booking->total_amount,
                'reference' => $booking->payment_reference,
                'callback_url' => route('payment.callback'),
                'metadata' => [
                    'booking_id' => $booking->id,
                    'booking_type' => 'car',
                    'user_id' => auth()->id(),
                ],
            ]);
            
            return redirect($result['payment_url']);
        } catch (\Exception $e) {
            return back()->withErrors(['payment' => $e->getMessage()]);
        }
    }

    public function showBooking(CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        $booking->load(['car.category', 'car.vendor', 'user']);

        return view('cars.booking-show', compact('booking'));
    }

    public function bankTransferInstructions(CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        $booking->load(['car.category', 'car.vendor']);

        return view('cars.bank-transfer', compact('booking'));
    }

    public function uploadPaymentProof(Request $request, CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        $validated = $request->validate([
            'payment_proof' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'payment_notes' => 'nullable|string|max:1000',
        ]);

        // Upload payment proof
        $proofPath = $request->file('payment_proof')->store('bookings/payment-proofs', 'public');

        // Update booking
        $booking->update([
            'payment_proof' => $proofPath,
            'payment_status' => 'pending_verification',
        ]);

        return redirect()->route('cars.booking.show', $booking)
            ->with('success', 'Payment proof uploaded successfully! We will verify and confirm your booking soon.');
    }

    public function confirmPickup(CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Verify booking is confirmed and paid
        if ($booking->status !== 'confirmed' || $booking->payment_status !== 'paid') {
            return back()->withErrors(['error' => 'Booking must be confirmed and paid before pickup.']);
        }

        // Update booking status
        $booking->update([
            'status' => 'active',
            'pickup_at' => now(),
            'deposit_status' => 'held', // Security deposit is now held
        ]);

        return redirect()->route('cars.booking.show', $booking)
            ->with('success', 'Vehicle pickup confirmed! Enjoy your rental. Remember to confirm return when you bring the vehicle back.');
    }

    public function confirmReturn(CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Verify booking is active
        if ($booking->status !== 'active') {
            return back()->withErrors(['error' => 'Booking must be active to confirm return.']);
        }

        // Update booking status
        $booking->update([
            'status' => 'pending_inspection',
            'returned_at' => now(),
            'deposit_status' => 'pending_inspection', // Waiting for vendor inspection
        ]);

        return redirect()->route('cars.booking.show', $booking)
            ->with('success', 'Vehicle return confirmed! The vendor will inspect the vehicle. Your security deposit will be refunded if no damages are found.');
    }

    public function requestRefund(CarBooking $booking)
    {
        // Check ownership
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Check if refund is available based on deposit_status
        if (!in_array($booking->deposit_status, ['refunded', 'deducted', 'partially_deducted'])) {
            return back()->withErrors(['error' => 'Refund is not yet available for this booking. The vendor must approve the refund first.']);
        }

        // Check if already requested
        if (in_array($booking->refund_payout_status, ['requested', 'processing', 'paid'])) {
            return back()->withErrors(['error' => 'Refund has already been requested or processed.']);
        }

        // Check if there's an amount to refund
        if (!$booking->deposit_refund_amount || $booking->deposit_refund_amount <= 0) {
            return back()->withErrors(['error' => 'No refund amount available.']);
        }

        $booking->update([
            'refund_payout_status' => 'requested',
            'refund_requested_at' => now(),
        ]);

        return back()->with('success', 'Refund request submitted successfully! Admin will process your refund within 3-5 business days.');
    }
}