<?php

namespace App\Http\Controllers;

use App\Models\Property;
use App\Models\Booking;
use App\Models\PaymentMethod;
use App\Services\BookingService;
use App\Services\Payment\PaymentFactory;
use Illuminate\Http\Request;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class BookingController extends Controller
{
    use AuthorizesRequests;
    public function create(Property $property)
    {
        return view('bookings.create', compact('property'));
    }
    
    public function store(Request $request, Property $property, BookingService $bookingService)
    {
        $validated = $request->validate([
            'start_date' => 'required|date|after:today',
            'end_date' => 'required|date|after:start_date',
            'adults_count' => 'required|integer|min:1',
            'children_count' => 'required|integer|min:0',
            'trip_type' => 'required|in:normal,party',
            'discount_code' => 'nullable|string',
        ]);
        
        // Check availability
        if (!$property->isAvailable($validated['start_date'], $validated['end_date'])) {
            return back()->withErrors(['dates' => 'Property is not available for selected dates.']);
        }
        
        $booking = $bookingService->createBooking(auth()->user(), $property, $validated);
        
        if ($booking->requires_vendor_approval) {
            return redirect()->route('bookings.show', $booking)
                ->with('success', 'Booking request sent! Waiting for vendor approval.');
        }
        
        // Redirect to payment
        return redirect()->route('bookings.payment', $booking);
    }
    
    public function show(Booking $booking)
    {
        $this->authorize('view', $booking);
        
        $booking->load(['property.vendor.user', 'user', 'cautionRefundLogs.actor']);
        
        return view('bookings.show', compact('booking'));
    }
    
    public function payment(Booking $booking)
    {
        $this->authorize('view', $booking);
        
        if ($booking->payment_status === 'paid') {
            return redirect()->route('bookings.show', $booking)
                ->with('info', 'This booking has already been paid.');
        }
        
        if ($booking->requires_vendor_approval && $booking->vendor_approval_status !== 'approved') {
            return redirect()->route('bookings.show', $booking)
                ->with('error', 'Please wait for vendor approval before making payment.');
        }
        
        // Show payment method selection page
        $availablePaymentMethods = $this->getAvailablePaymentMethods();
        
        return view('bookings.payment', compact('booking', 'availablePaymentMethods'));
    }
    
    public function processPayment(Request $request, Booking $booking)
    {
        $this->authorize('view', $booking);
        
        // Get active payment method slugs for validation
        $activePaymentMethods = PaymentMethod::active()->pluck('slug')->toArray();
        
        $validated = $request->validate([
            'payment_method' => 'required|in:' . implode(',', $activePaymentMethods),
        ]);
        
        // Update booking with selected payment method
        $booking->update(['payment_method' => $validated['payment_method']]);
        
        // Handle bank transfer differently - no gateway needed
        if ($validated['payment_method'] === 'bank_transfer') {
            // Update booking status to awaiting payment confirmation
            $booking->update([
                'payment_status' => 'pending',
                'booking_status' => 'pending_payment',
            ]);
            
            return redirect()->route('bookings.bank-transfer-instructions', $booking)
                ->with('success', 'Please complete the bank transfer and upload proof of payment.');
        }
        
        // Initialize payment for online gateways (Paystack, Flutterwave)
        $paymentService = PaymentFactory::create($validated['payment_method']);
        
        try {
            $result = $paymentService->initializePayment([
                'email' => auth()->user()->email,
                'amount' => $booking->total_amount,
                'reference' => $booking->payment_reference,
                'callback_url' => route('payment.callback'),
                'metadata' => [
                    'booking_id' => $booking->id,
                    'user_id' => auth()->id(),
                ],
            ]);
            
            return redirect($result['payment_url']);
        } catch (\Exception $e) {
            return back()->withErrors(['payment' => $e->getMessage()]);
        }
    }
    
    protected function getAvailablePaymentMethods(): array
    {
        $methods = PaymentMethod::active()->get();
        
        return $methods->map(function($method) {
            return [
                'id' => $method->slug,
                'name' => $method->name,
                'description' => $method->description,
                'settings' => $method->settings,
            ];
        })->toArray();
    }
    
    public function bankTransferInstructions(Booking $booking)
    {
        $this->authorize('view', $booking);
        
        return view('bookings.bank-transfer', compact('booking'));
    }
    
    public function uploadPaymentProof(Request $request, Booking $booking)
    {
        $this->authorize('view', $booking);
        
        $validated = $request->validate([
            'payment_proof' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'payment_notes' => 'nullable|string|max:500',
        ]);
        
        // Upload proof
        if ($request->hasFile('payment_proof')) {
            $path = $request->file('payment_proof')->store('payment-proofs', 'public');
            
            $booking->update([
                'payment_proof' => $path,
                'payment_notes' => $validated['payment_notes'] ?? null,
                'payment_status' => 'pending_verification',
            ]);
        }
        
        return redirect()->route('bookings.show', $booking)
            ->with('success', 'Payment proof uploaded successfully! We will verify and confirm your booking soon.');
    }
    
    public function index()
    {
        $bookings = auth()->user()
            ->bookings()
            ->with(['property', 'vendor'])
            ->latest()
            ->paginate(10, ['*'], 'properties_page');
        
        $carBookings = auth()->user()
            ->carBookings()
            ->with(['car.category', 'vendor'])
            ->latest()
            ->paginate(10, ['*'], 'cars_page');
        
        return view('bookings.index', compact('bookings', 'carBookings'));
    }

    public function complete(Booking $booking)
    {
        $this->authorize('view', $booking);

        // Check if booking is confirmed and checkout date has passed
        if ($booking->booking_status !== 'confirmed') {
            return back()->withErrors(['error' => 'Only confirmed bookings can be marked as completed.']);
        }

        if (!$booking->end_date->isPast()) {
            return back()->withErrors(['error' => 'Booking can only be completed after the checkout date.']);
        }

        // Mark as completed
        $booking->update([
            'booking_status' => 'completed',
        ]);

        return back()->with('success', 'Your stay has been marked as completed! You can now write a review and await your caution fee refund.');
    }

    public function confirmCheckIn(Booking $booking)
    {
        $this->authorize('view', $booking);

        // Check if booking is confirmed and payment is made
        if ($booking->booking_status !== 'confirmed' || $booking->payment_status !== 'paid') {
            return back()->withErrors(['error' => 'Booking must be confirmed and paid before check-in.']);
        }

        // Check if already checked in
        if ($booking->checked_in_at) {
            return back()->with('info', 'You have already checked in.');
        }

        // Check if check-in date has arrived
        if (now()->lt($booking->start_date)) {
            return back()->withErrors(['error' => 'Check-in date has not arrived yet.']);
        }

        $booking->update([
            'checked_in_at' => now(),
            'booking_status' => 'checked_in',
        ]);

        // Log activity
        \App\Helpers\ActivityLogger::log(
            'user_checkin',
            "User checked in for Booking #{$booking->id}",
            $booking,
            [
                'user_id' => auth()->id(),
                'user_name' => auth()->user()->name,
                'property' => $booking->property->title,
                'checked_in_at' => now()->format('Y-m-d H:i:s')
            ]
        );

        return back()->with('success', 'Check-in confirmed! Enjoy your stay.');
    }

    public function confirmCheckOut(Booking $booking)
    {
        $this->authorize('view', $booking);

        // Check if checked in (either timestamp or status)
        if (!$booking->checked_in_at && $booking->booking_status !== 'checked_in') {
            return back()->withErrors(['error' => 'You must check in first before checking out.']);
        }

        // Check if already checked out
        if ($booking->checked_out_at) {
            return back()->with('info', 'You have already checked out.');
        }

        $booking->update([
            'checked_out_at' => now(),
            'booking_status' => 'completed',
        ]);

        // Log activity
        \App\Helpers\ActivityLogger::log(
            'user_checkout',
            "User checked out from Booking #{$booking->id}",
            $booking,
            [
                'user_id' => auth()->id(),
                'user_name' => auth()->user()->name,
                'property' => $booking->property->title,
                'checked_out_at' => now()->format('Y-m-d H:i:s'),
                'stay_duration' => $booking->nights_count . ' nights'
            ]
        );

        return back()->with('success', 'Check-out confirmed! Thank you for your stay. Your caution fee will be processed by the vendor.');
    }

    public function requestRefund(Booking $booking)
    {
        // Ensure user owns this booking
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Check if refund is available
        if (!in_array($booking->caution_refund_status, ['refunded', 'partially_deducted', 'deducted'])) {
            return back()->withErrors(['error' => 'Refund is not yet available for this booking.']);
        }

        // Check if already requested
        if ($booking->refund_payout_status !== 'pending') {
            return back()->withErrors(['error' => 'Refund has already been requested.']);
        }

        // Check if there's an amount to refund
        if (!$booking->caution_refund_amount || $booking->caution_refund_amount <= 0) {
            return back()->withErrors(['error' => 'No refund amount available.']);
        }

        $booking->update([
            'refund_payout_status' => 'requested',
            'refund_requested_at' => now(),
        ]);

        // Log activity
        \App\Helpers\ActivityLogger::log(
            'refund_requested',
            "User requested caution fee refund for Booking #{$booking->id}",
            $booking,
            [
                'user_id' => auth()->id(),
                'user_name' => auth()->user()->name,
                'refund_amount' => $booking->caution_refund_amount,
                'requested_at' => now()->format('Y-m-d H:i:s')
            ]
        );

        return back()->with('success', 'Refund request submitted successfully! Admin will process your refund within 3-5 business days.');
    }

    public function downloadReceipt(Booking $booking)
    {
        // Ensure user owns this booking
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        // Ensure payment is completed
        if ($booking->payment_status !== 'paid') {
            return back()->withErrors(['error' => 'Receipt is only available for paid bookings.']);
        }

        return view('bookings.receipt', compact('booking'));
    }
}
