<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Property;
use Illuminate\Http\Request;

class PropertyController extends Controller
{
    public function index(Request $request)
    {
        $query = Property::query()
            ->with(['vendor.user', 'reviews'])
            ->active();

        // Filters
        if ($request->filled('city')) {
            $query->inCity($request->city);
        }

        if ($request->filled('min_price') && $request->filled('max_price')) {
            $query->inPriceRange($request->min_price, $request->max_price);
        }

        if ($request->filled('guests')) {
            $query->where('max_guests', '>=', $request->guests);
        }

        // Featured properties first
        $query->orderByRaw('featured_until IS NOT NULL AND featured_until >= NOW() DESC');
        $query->orderBy('created_at', 'desc');

        $properties = $query->paginate(20);

        return response()->json([
            'data' => $properties->items(),
            'meta' => [
                'current_page' => $properties->currentPage(),
                'last_page' => $properties->lastPage(),
                'per_page' => $properties->perPage(),
                'total' => $properties->total(),
            ],
        ]);
    }

    public function show(Property $property)
    {
        $property->load(['vendor.user', 'reviews.user']);

        return response()->json([
            'data' => $property,
            'average_rating' => $property->averageRating(),
        ]);
    }

    public function bookedDates(Property $property)
    {
        // Get all confirmed and paid bookings for this property
        $bookings = $property->bookings()
            ->where('payment_status', 'paid')
            ->whereIn('booking_status', ['confirmed', 'pending'])
            ->get();

        $bookedDates = [];

        // Add dates from bookings
        foreach ($bookings as $booking) {
            $checkIn = new \DateTime($booking->check_in_date);
            $checkOut = new \DateTime($booking->check_out_date);
            
            // Include all dates from check-in to check-out
            $interval = new \DateInterval('P1D');
            $dateRange = new \DatePeriod($checkIn, $interval, $checkOut);

            foreach ($dateRange as $date) {
                $bookedDates[] = $date->format('Y-m-d');
            }
            
            // Include check-out date as well
            $bookedDates[] = $checkOut->format('Y-m-d');
        }

        // Add vendor-marked unavailable dates
        if (!empty($property->unavailable_dates) && is_array($property->unavailable_dates)) {
            foreach ($property->unavailable_dates as $unavailableDate) {
                $bookedDates[] = date('Y-m-d', strtotime($unavailableDate));
            }
        }

        // Remove duplicates and sort
        $bookedDates = array_unique($bookedDates);
        sort($bookedDates);

        return response()->json([
            'booked_dates' => array_values($bookedDates),
        ]);
    }
}