<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VendorAd;
use Illuminate\Http\Request;

class VendorAdController extends Controller
{
    public function index()
    {
        // Get ads with payment proof pending verification (bank transfer)
        $pendingPaymentAds = VendorAd::with(['vendor', 'property', 'adPlan'])
            ->where('payment_status', 'pending')
            ->whereNotNull('payment_proof')
            ->latest()
            ->get();

        // Get paid ads pending admin approval
        $pendingAds = VendorAd::with(['vendor', 'property', 'adPlan'])
            ->where('payment_status', 'paid')
            ->where('approved_by_admin', false)
            ->latest()
            ->get();

        // Get active ads
        $activeAds = VendorAd::with(['vendor', 'property', 'adPlan'])
            ->where('active', true)
            ->latest()
            ->get();

        return view('admin.ads.index', compact('pendingPaymentAds', 'pendingAds', 'activeAds'));
    }

    public function verifyPayment(VendorAd $vendorAd)
    {
        // Mark payment as verified
        $vendorAd->update([
            'payment_status' => 'paid',
            'paid_at' => now(),
        ]);

        // Track promo code usage if promo code was applied
        if ($vendorAd->promo_code_id) {
            $promoCodeService = app(\App\Services\PromoCodeService::class);
            $promoCode = \App\Models\PromoCode::find($vendorAd->promo_code_id);
            
            if ($promoCode) {
                $promoCodeService->applyPromoCode(
                    $promoCode,
                    $vendorAd->vendor,
                    $vendorAd,
                    $vendorAd->adPlan->price,
                    $vendorAd->discount_amount,
                    $vendorAd->final_amount
                );
            }
        }

        return back()->with('success', 'Payment verified successfully. Ad is now pending approval.');
    }

    public function rejectPayment(Request $request, VendorAd $vendorAd)
    {
        $validated = $request->validate([
            'reason' => 'nullable|string|max:500',
        ]);

        // Delete the payment proof and keep status as pending
        if ($vendorAd->payment_proof) {
            \Storage::disk('public')->delete($vendorAd->payment_proof);
        }

        $vendorAd->update([
            'payment_proof' => null,
        ]);

        // TODO: Notify vendor about rejection with reason

        return back()->with('success', 'Payment proof rejected. Vendor will be notified.');
    }

    public function approve(VendorAd $vendorAd)
    {
        $endDate = now()->addDays($vendorAd->adPlan->duration_days);
        
        $vendorAd->update([
            'approved_by_admin' => true,
            'approved_by' => auth()->id(),
            'approved_at' => now(),
            'start_date' => now(),
            'end_date' => $endDate,
            'active' => true,
        ]);

        // Also set the property as featured for the ad duration
        $vendorAd->property->update([
            'featured_until' => $endDate,
        ]);

        return back()->with('success', 'Advertisement approved and activated. Property is now featured on the home page.');
    }

    public function deactivate(VendorAd $vendorAd)
    {
        $vendorAd->update(['active' => false]);

        return back()->with('success', 'Advertisement deactivated.');
    }

    public function extend(Request $request, VendorAd $vendorAd)
    {
        $validated = $request->validate([
            'days' => 'required|integer|min:1|max:90',
        ]);

        $vendorAd->update([
            'end_date' => $vendorAd->end_date->addDays($validated['days']),
        ]);

        return back()->with('success', "Advertisement extended by {$validated['days']} days.");
    }
}
