<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function index()
    {
        $users = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['owner', 'manager', 'moderator']);
        })->with('roles')->latest()->paginate(20);
        
        $roles = Role::whereIn('name', ['manager', 'moderator'])->get();
        
        return view('admin.users.index', compact('users', 'roles'));
    }
    
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|max:50',
            'password' => ['required', 'confirmed', Password::defaults()],
            'role' => 'required|in:manager,moderator',
        ]);
        
        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'email_verified_at' => now(), // Auto-verify admin users
        ]);
        
        $user->assignRole($validated['role']);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'create_admin_user',
            "Created {$validated['role']} account for {$user->name}",
            $user
        );
        
        return back()->with('success', ucfirst($validated['role']) . ' account created successfully.');
    }
    
    public function update(Request $request, User $user)
    {
        // Prevent modifying owner accounts
        if ($user->hasRole('owner')) {
            return back()->withErrors(['error' => 'Cannot modify owner accounts.']);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string|max:50',
            'role' => 'required|in:manager,moderator',
            'password' => ['nullable', 'confirmed', Password::defaults()],
        ]);
        
        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
        ]);
        
        if (!empty($validated['password'])) {
            $user->update(['password' => Hash::make($validated['password'])]);
        }
        
        // Update role
        $user->syncRoles([$validated['role']]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'update_admin_user',
            "Updated admin user: {$user->name}",
            $user
        );
        
        return back()->with('success', 'User updated successfully.');
    }
    
    public function destroy(User $user)
    {
        // Prevent deleting owner accounts or self
        if ($user->hasRole('owner')) {
            return back()->withErrors(['error' => 'Cannot delete owner accounts.']);
        }
        
        if ($user->id === auth()->id()) {
            return back()->withErrors(['error' => 'Cannot delete your own account.']);
        }
        
        $userName = $user->name;
        $user->delete();
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'delete_admin_user',
            "Deleted admin user: {$userName}",
            null,
            ['user_id' => $user->id, 'user_name' => $userName]
        );
        
        return back()->with('success', 'User deleted successfully.');
    }
    
    public function toggleStatus(User $user)
    {
        // Prevent modifying owner accounts or self
        if ($user->hasRole('owner') || $user->id === auth()->id()) {
            return back()->withErrors(['error' => 'Cannot modify this account.']);
        }
        
        // Toggle by updating email_verified_at (null = disabled)
        $user->update([
            'email_verified_at' => $user->email_verified_at ? null : now(),
        ]);
        
        $status = $user->email_verified_at ? 'enabled' : 'disabled';
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'toggle_admin_user_status',
            "Changed admin user status to {$status}: {$user->name}",
            $user
        );
        
        return back()->with('success', "User account {$status} successfully.");
    }
}
