<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;

class SettingsController extends Controller
{
    public function index()
    {
        $settings = [
            'site_name' => Setting::get('site_name', 'ShortStayNG'),
            'site_description' => Setting::get('site_description', ''),
            'logo' => Setting::get('logo', ''),
            'favicon' => Setting::get('favicon', ''),
            'cac_certificate' => Setting::get('cac_certificate', ''),
            'global_commission_percentage' => Setting::get('global_commission_percentage', 15),
            'contact_email' => Setting::get('contact_email', ''),
            'contact_phone' => Setting::get('contact_phone', ''),
            'whatsapp_number' => Setting::get('whatsapp_number', ''),
            'contact_address' => Setting::get('contact_address', ''),
            'facebook_url' => Setting::get('facebook_url', ''),
            'instagram_url' => Setting::get('instagram_url', ''),
            'linkedin_url' => Setting::get('linkedin_url', ''),
            'youtube_url' => Setting::get('youtube_url', ''),
            'twitter_url' => Setting::get('twitter_url', ''),
            'tiktok_url' => Setting::get('tiktok_url', ''),
            'enable_user_registration' => Setting::get('enable_user_registration', true),
            'enable_vendor_registration' => Setting::get('enable_vendor_registration', true),
            // Car Rental Settings
            'car_rental_enabled' => Setting::get('car_rental_enabled', false),
            'car_rental_commission_rate' => Setting::get('car_rental_commission_rate', 15),
            'car_rental_minimum_age' => Setting::get('car_rental_minimum_age', 21),
            'car_rental_minimum_license_years' => Setting::get('car_rental_minimum_license_years', 1),
            'car_rental_terms' => Setting::get('car_rental_terms', ''),
        ];
        
        return view('admin.settings', compact('settings'));
    }
    
    public function update(Request $request)
    {
        $validated = $request->validate([
            'site_name' => 'required|string|max:255',
            'site_description' => 'nullable|string',
            'global_commission_percentage' => 'required|numeric|min:0|max:100',
            'contact_email' => 'nullable|email',
            'contact_phone' => 'nullable|string|max:50',
            'whatsapp_number' => 'nullable|string|max:50',
            'contact_address' => 'nullable|string',
            'facebook_url' => 'nullable|url',
            'instagram_url' => 'nullable|url',
            'linkedin_url' => 'nullable|url',
            'youtube_url' => 'nullable|url',
            'twitter_url' => 'nullable|url',
            'tiktok_url' => 'nullable|url',
            'logo' => 'nullable|image|max:2048',
            'favicon' => 'nullable|image|max:1024',
            'cac_certificate' => 'nullable|file|mimes:jpeg,png,jpg,pdf|max:5120',
            // Car Rental Settings
            'car_rental_commission_rate' => 'required|numeric|min:0|max:100',
            'car_rental_minimum_age' => 'required|integer|min:18|max:100',
            'car_rental_minimum_license_years' => 'required|integer|min:0|max:50',
            'car_rental_terms' => 'nullable|string',
        ]);
        
        // Handle checkboxes (they don't send values when unchecked)
        $enableUserRegistration = $request->has('enable_user_registration');
        $enableVendorRegistration = $request->has('enable_vendor_registration');
        $carRentalEnabled = $request->has('car_rental_enabled');
        
        // Save registration settings as boolean
        Setting::set('enable_user_registration', $enableUserRegistration, 'boolean');
        Setting::set('enable_vendor_registration', $enableVendorRegistration, 'boolean');
        Setting::set('car_rental_enabled', $carRentalEnabled, 'boolean');
        
        foreach ($validated as $key => $value) {
            if ($key === 'logo' || $key === 'favicon' || $key === 'cac_certificate') {
                if ($request->hasFile($key)) {
                    $path = $request->file($key)->store('settings', 'public');
                    Setting::set($key, $path);
                }
            } else {
                $type = is_numeric($value) ? 'number' : 'string';
                Setting::set($key, $value, $type);
            }
        }
        
        return back()->with('success', 'Settings updated successfully.');
    }
}