<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Review;
use Illuminate\Http\Request;

class ReviewController extends Controller
{
    public function index(Request $request)
    {
        $query = Review::with(['user', 'property', 'booking']);

        if ($request->filled('visible')) {
            $query->where('visible', $request->boolean('visible'));
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->whereHas('user', function ($userQuery) use ($search) {
                    $userQuery->where('name', 'like', "%{$search}%");
                })->orWhereHas('property', function ($propertyQuery) use ($search) {
                    $propertyQuery->where('title', 'like', "%{$search}%");
                });
            });
        }

        $reviews = $query->latest()->paginate(20);

        return view('admin.reviews.index', compact('reviews'));
    }

    public function moderate(Request $request, Review $review)
    {
        $validated = $request->validate([
            'action' => 'required|in:hide,show,delete',
            'reason' => 'nullable|string|max:500',
        ]);

        switch ($validated['action']) {
            case 'hide':
                $review->update([
                    'visible' => false,
                    'moderated' => true,
                    'moderated_by' => auth()->id(),
                    'moderated_at' => now(),
                    'moderation_reason' => $validated['reason'] ?? null,
                ]);
                $message = 'Review hidden successfully.';
                break;

            case 'show':
                $review->update([
                    'visible' => true,
                    'moderated' => true,
                    'moderated_by' => auth()->id(),
                    'moderated_at' => now(),
                    'moderation_reason' => null,
                ]);
                $message = 'Review made visible.';
                break;

            case 'delete':
                $review->delete();
                $message = 'Review deleted successfully.';
                break;
        }

        return back()->with('success', $message);
    }
}
