<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\Vendor;
use App\Services\Maps\GoogleMapsService;
use Illuminate\Http\Request;

class PropertyController extends Controller
{
    public function create()
    {
        // Get all verified vendors or create a default "Company" vendor
        $vendors = Vendor::where('verified', true)->get();
        
        // If no company vendor exists, we'll need to create one or use a default
        $companyVendor = Vendor::where('company_name', 'ShortStayNG')->first();
        
        return view('admin.properties.create', compact('vendors', 'companyVendor'));
    }
    
    public function store(Request $request, GoogleMapsService $mapsService)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price_per_night' => 'required|numeric|min:0',
            'caution_fee' => 'required|numeric|min:0',
            'allow_parties' => 'boolean',
            'address' => 'required|string',
            'city' => 'required|string',
            'state' => 'required|string',
            'house_type' => 'required|string',
            'bedrooms' => 'required|integer|min:0',
            'beds' => 'required|integer|min:0',
            'bathrooms' => 'required|integer|min:0',
            'max_guests' => 'required|integer|min:1',
            'check_in_from' => 'required|date_format:H:i',
            'check_in_until' => 'required|date_format:H:i',
            'check_out_time' => 'required|date_format:H:i',
            'highlighted_features' => 'nullable|array',
            'amenities' => 'nullable|array',
            'house_rules' => 'nullable|array',
            'photos' => 'nullable|array',
            'photos.*' => 'image|max:5120',
            'location_lat' => 'nullable|numeric',
            'location_lng' => 'nullable|numeric',
            'vendor_id' => 'required|exists:vendors,id',
        ]);
        
        // Geocode address
        $fullAddress = "{$validated['address']}, {$validated['city']}, {$validated['state']}, Nigeria";
        $coordinates = $mapsService->geocodeAddress($fullAddress);
        
        if ($coordinates) {
            $validated['location_lat'] = $coordinates['latitude'];
            $validated['location_lng'] = $coordinates['longitude'];
        }
        
        // Handle photo uploads
        if ($request->hasFile('photos')) {
            $photoPaths = [];
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('properties', 'public');
                $photoPaths[] = $path;
            }
            $validated['photos'] = $photoPaths;
        }
        
        // Admin-created properties are auto-approved
        $validated['status'] = 'approved';
        $validated['approved_at'] = now();
        $validated['approved_by'] = auth()->id();
        
        $property = Property::create($validated);
        
        return redirect()->route('admin.properties.show', $property)
            ->with('success', 'Property created successfully and is now live!');
    }
    
    public function edit(Property $property)
    {
        $vendors = Vendor::where('verified', true)->get();
        $companyVendor = Vendor::where('company_name', 'ShortStayNG')->first();
        
        return view('admin.properties.edit', compact('property', 'vendors', 'companyVendor'));
    }
    
    public function update(Request $request, Property $property, GoogleMapsService $mapsService)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price_per_night' => 'required|numeric|min:0',
            'caution_fee' => 'required|numeric|min:0',
            'allow_parties' => 'boolean',
            'address' => 'required|string',
            'city' => 'required|string',
            'state' => 'required|string',
            'house_type' => 'required|string',
            'bedrooms' => 'required|integer|min:0',
            'beds' => 'required|integer|min:0',
            'bathrooms' => 'required|integer|min:0',
            'max_guests' => 'required|integer|min:1',
            'check_in_from' => 'required|date_format:H:i',
            'check_in_until' => 'required|date_format:H:i',
            'check_out_time' => 'required|date_format:H:i',
            'highlighted_features' => 'nullable|array',
            'amenities' => 'nullable|array',
            'house_rules' => 'nullable|array',
            'photos' => 'nullable|array',
            'photos.*' => 'image|max:5120',
            'location_lat' => 'nullable|numeric',
            'location_lng' => 'nullable|numeric',
            'vendor_id' => 'required|exists:vendors,id',
            'delete_photos' => 'nullable|array',
        ]);
        
        // Geocode address if changed
        $fullAddress = "{$validated['address']}, {$validated['city']}, {$validated['state']}, Nigeria";
        if ($property->address !== $validated['address'] || 
            $property->city !== $validated['city'] || 
            $property->state !== $validated['state']) {
            $coordinates = $mapsService->geocodeAddress($fullAddress);
            
            if ($coordinates) {
                $validated['location_lat'] = $coordinates['latitude'];
                $validated['location_lng'] = $coordinates['longitude'];
            }
        }
        
        // Handle photo deletions
        if ($request->has('delete_photos')) {
            $existingPhotos = $property->photos ?? [];
            $photosToDelete = $request->input('delete_photos');
            
            foreach ($photosToDelete as $photoPath) {
                if (in_array($photoPath, $existingPhotos)) {
                    \Storage::disk('public')->delete($photoPath);
                }
            }
            
            $validated['photos'] = array_values(array_diff($existingPhotos, $photosToDelete));
        } else {
            $validated['photos'] = $property->photos ?? [];
        }
        
        // Handle new photo uploads
        if ($request->hasFile('photos')) {
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('properties', 'public');
                $validated['photos'][] = $path;
            }
        }
        
        $property->update($validated);
        
        return redirect()->route('admin.properties.show', $property)
            ->with('success', 'Property updated successfully!');
    }
    
    public function destroy(Property $property)
    {
        // Check if property has active bookings
        $activeBookings = $property->bookings()
            ->whereIn('booking_status', ['pending', 'confirmed'])
            ->count();
        
        if ($activeBookings > 0) {
            return back()->with('error', 'Cannot delete property with active bookings.');
        }
        
        // Delete photos
        if ($property->photos) {
            foreach ($property->photos as $photo) {
                \Storage::disk('public')->delete($photo);
            }
        }
        
        $property->delete();
        
        return redirect()->route('admin.properties.index')
            ->with('success', 'Property deleted successfully!');
    }
}
