<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Property;
use Illuminate\Http\Request;

class PropertyApprovalController extends Controller
{
    public function index(Request $request)
    {
        $query = Property::with(['vendor.user']);
        
        $status = $request->get('status', 'pending');
        
        if ($status) {
            $query->where('status', $status);
        }
        
        $properties = $query->latest()->paginate(20);
        
        return view('admin.properties.index', compact('properties', 'status'));
    }
    
    public function show(Property $property)
    {
        $property->load(['vendor.user', 'bookings']);
        
        return view('admin.properties.show', compact('property'));
    }
    
    public function approve(Property $property)
    {
        $property->update([
            'status' => 'active',
            'rejection_reason' => null,
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'approve_property',
            "Approved property: {$property->title}",
            $property
        );
        
        // TODO: Send notification to vendor
        
        return back()->with('success', 'Property approved successfully.');
    }
    
    public function reject(Request $request, Property $property)
    {
        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:1000',
        ]);
        
        $property->update([
            'status' => 'rejected',
            'rejection_reason' => $validated['rejection_reason'],
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'reject_property',
            "Rejected property: {$property->title}",
            $property,
            ['reason' => $validated['rejection_reason']]
        );
        
        // TODO: Send notification to vendor
        
        return back()->with('success', 'Property rejected.');
    }
    
    public function toggleStatus(Property $property)
    {
        $newStatus = $property->status === 'active' ? 'inactive' : 'active';
        
        $property->update(['status' => $newStatus]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'toggle_property_status',
            "Changed property status to {$newStatus}: {$property->title}",
            $property
        );
        
        return back()->with('success', 'Property status updated.');
    }
}
