<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PromoCode;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PromoCodeController extends Controller
{
    public function index()
    {
        $promoCodes = PromoCode::with('creator')
            ->withCount('usages')
            ->latest()
            ->paginate(20);
        
        return view('admin.promo-codes.index', compact('promoCodes'));
    }
    
    public function create()
    {
        return view('admin.promo-codes.create');
    }
    
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:promo_codes,code|alpha_dash',
            'description' => 'nullable|string',
            'discount_type' => 'required|in:percentage,fixed',
            'discount_value' => 'required|numeric|min:0',
            'max_discount_amount' => 'nullable|numeric|min:0',
            'min_ad_amount' => 'required|numeric|min:0',
            'usage_limit' => 'nullable|integer|min:1',
            'usage_per_vendor' => 'required|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date|after:valid_from',
            'is_active' => 'boolean',
        ]);
        
        // Convert code to uppercase
        $validated['code'] = strtoupper($validated['code']);
        $validated['created_by'] = auth()->id();
        
        // Validate percentage value
        if ($validated['discount_type'] === 'percentage' && $validated['discount_value'] > 100) {
            return back()->withErrors(['discount_value' => 'Percentage cannot exceed 100%'])->withInput();
        }
        
        PromoCode::create($validated);
        
        return redirect()->route('admin.promo-codes.index')
            ->with('success', 'Promo code created successfully!');
    }
    
    public function edit(PromoCode $promoCode)
    {
        return view('admin.promo-codes.edit', compact('promoCode'));
    }
    
    public function update(Request $request, PromoCode $promoCode)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|alpha_dash|unique:promo_codes,code,' . $promoCode->id,
            'description' => 'nullable|string',
            'discount_type' => 'required|in:percentage,fixed',
            'discount_value' => 'required|numeric|min:0',
            'max_discount_amount' => 'nullable|numeric|min:0',
            'min_ad_amount' => 'required|numeric|min:0',
            'usage_limit' => 'nullable|integer|min:1',
            'usage_per_vendor' => 'required|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date|after:valid_from',
            'is_active' => 'boolean',
        ]);
        
        // Convert code to uppercase
        $validated['code'] = strtoupper($validated['code']);
        
        // Validate percentage value
        if ($validated['discount_type'] === 'percentage' && $validated['discount_value'] > 100) {
            return back()->withErrors(['discount_value' => 'Percentage cannot exceed 100%'])->withInput();
        }
        
        $promoCode->update($validated);
        
        return redirect()->route('admin.promo-codes.index')
            ->with('success', 'Promo code updated successfully!');
    }
    
    public function destroy(PromoCode $promoCode)
    {
        // Check if promo code has been used
        if ($promoCode->usage_count > 0) {
            return back()->with('error', 'Cannot delete promo code that has been used. Deactivate it instead.');
        }
        
        $promoCode->delete();
        
        return redirect()->route('admin.promo-codes.index')
            ->with('success', 'Promo code deleted successfully!');
    }
    
    public function toggle(PromoCode $promoCode)
    {
        $promoCode->update(['is_active' => !$promoCode->is_active]);
        
        $status = $promoCode->is_active ? 'activated' : 'deactivated';
        
        return back()->with('success', "Promo code {$status} successfully!");
    }
}
