<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Payout;
use Illuminate\Http\Request;

class PayoutController extends Controller
{
    public function index(Request $request)
    {
        $status = $request->get('status', 'pending');
        
        $payouts = Payout::with(['vendor.user'])
            ->where('status', $status)
            ->latest()
            ->paginate(20);
        
        return view('admin.payouts.index', compact('payouts', 'status'));
    }
    
    public function show(Payout $payout)
    {
        $payout->load(['vendor.user']);
        
        return view('admin.payouts.show', compact('payout'));
    }
    
    public function approve(Request $request, Payout $payout)
    {
        $validated = $request->validate([
            'transaction_reference' => 'nullable|string|max:255',
            'notes' => 'nullable|string|max:1000',
        ]);
        
        $payout->update([
            'status' => 'completed',
            'processed_at' => now(),
            'processed_by' => auth()->id(),
            'transaction_reference' => $validated['transaction_reference'] ?? null,
            'admin_notes' => $validated['notes'] ?? null,
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'approve_payout',
            "Approved payout of ₦" . number_format($payout->amount) . " for {$payout->vendor->company_name}",
            $payout,
            [
                'amount' => $payout->amount,
                'transaction_reference' => $validated['transaction_reference'] ?? null,
            ]
        );
        
        // TODO: Send notification to vendor
        
        return back()->with('success', 'Payout marked as completed successfully.');
    }
    
    public function reject(Request $request, Payout $payout)
    {
        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:1000',
        ]);
        
        $payout->update([
            'status' => 'rejected',
            'processed_at' => now(),
            'processed_by' => auth()->id(),
            'rejection_reason' => $validated['rejection_reason'],
        ]);
        
        // Refund the amount to vendor's pending balance
        $payout->vendor->increment('pending_balance', $payout->amount);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'reject_payout',
            "Rejected payout of ₦" . number_format($payout->amount) . " for {$payout->vendor->company_name}",
            $payout,
            ['reason' => $validated['rejection_reason']]
        );
        
        // TODO: Send notification to vendor
        
        return back()->with('success', 'Payout rejected and amount refunded to vendor balance.');
    }
}
