<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use Illuminate\Http\Request;

class PaymentVerificationController extends Controller
{
    public function index(Request $request)
    {
        $status = $request->get('status', 'pending_verification');
        
        $bookings = Booking::with(['user', 'property', 'vendor'])
            ->where('payment_method', 'bank_transfer')
            ->where('payment_status', $status)
            ->latest()
            ->paginate(20);
        
        return view('admin.payment-verification.index', compact('bookings', 'status'));
    }
    
    public function show(Booking $booking)
    {
        $booking->load(['user', 'property', 'vendor']);
        
        return view('admin.payment-verification.show', compact('booking'));
    }
    
    public function approve(Request $request, Booking $booking)
    {
        $booking->update([
            'payment_status' => 'paid',
            'booking_status' => 'confirmed',
            'admin_verified' => true,
            'verified_by' => auth()->id(),
            'verified_at' => now(),
            'paid_at' => now(),
            'user_visible_vendor_info' => true,
            'vendor_visible_user_info' => true,
            'map_visible' => true,
        ]);
        
        // Track discount deal usage if discount was applied
        if ($booking->discount_deal_id) {
            $discountService = app(\App\Services\DiscountDealService::class);
            $discountDeal = \App\Models\DiscountDeal::find($booking->discount_deal_id);
            
            if ($discountDeal) {
                $discountService->applyDiscountDeal(
                    $discountDeal,
                    $booking->user,
                    $booking,
                    $booking->subtotal + $booking->caution_fee_amount,
                    $booking->discount_amount,
                    $booking->total_amount
                );
            }
        }
        
        // Credit vendor balance
        $vendor = $booking->vendor;
        $vendor->increment('pending_balance', $booking->vendor_payout_amount);
        $vendor->increment('total_earned', $booking->vendor_payout_amount);
        
        // Cancel conflicting unpaid bookings if no units available
        $bookingService = app(\App\Services\BookingService::class);
        $cancelledCount = $bookingService->cancelConflictingUnpaidBookings($booking);
        
        if ($cancelledCount > 0) {
            \Log::info('Auto-cancelled conflicting bookings', [
                'paid_booking_id' => $booking->id,
                'cancelled_count' => $cancelledCount,
            ]);
        }
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'approve_payment',
            "Approved bank transfer payment of ₦" . number_format($booking->total_amount) . " for Booking #{$booking->id}",
            $booking,
            [
                'amount' => $booking->total_amount,
                'vendor_payout' => $booking->vendor_payout_amount,
                'property' => $booking->property->title,
            ]
        );
        
        // TODO: Send notification to user and vendor
        
        $message = 'Payment verified and booking confirmed successfully. Vendor balance updated.';
        if ($cancelledCount > 0) {
            $message .= " {$cancelledCount} conflicting unpaid booking(s) were automatically cancelled.";
        }
        
        return back()->with('success', $message);
    }
    
    public function reject(Request $request, Booking $booking)
    {
        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:1000',
        ]);
        
        $booking->update([
            'payment_status' => 'failed',
            'booking_status' => 'cancelled',
            'cancellation_reason' => 'Payment verification failed: ' . $validated['rejection_reason'],
            'cancelled_at' => now(),
            'verified_by' => auth()->id(),
            'verified_at' => now(),
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'reject_payment',
            "Rejected bank transfer payment of ₦" . number_format($booking->total_amount) . " for Booking #{$booking->id}",
            $booking,
            ['reason' => $validated['rejection_reason']]
        );
        
        // TODO: Send notification to user
        
        return back()->with('success', 'Payment rejected and booking cancelled.');
    }
}
