<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;

class PaymentMethodController extends Controller
{
    public function index()
    {
        $paymentMethods = PaymentMethod::orderBy('display_order')->get();
        
        return view('admin.payment-methods.index', compact('paymentMethods'));
    }
    
    public function update(Request $request, PaymentMethod $paymentMethod)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'display_order' => 'required|integer|min:0',
        ]);
        
        // Handle bank transfer settings
        if ($paymentMethod->slug === 'bank_transfer') {
            $settings = $request->validate([
                'bank_name' => 'required|string|max:255',
                'account_name' => 'required|string|max:255',
                'account_number' => 'required|string|max:50',
            ]);
            
            $validated['settings'] = array_merge(
                $paymentMethod->settings ?? [],
                $settings
            );
        }
        
        $paymentMethod->update($validated);
        
        return back()->with('success', 'Payment method updated successfully.');
    }
    
    public function toggleStatus(PaymentMethod $paymentMethod)
    {
        // Check if method can be activated
        if (!$paymentMethod->is_active) {
            if ($paymentMethod->slug === 'paystack') {
                if (empty(config('services.paystack.secret_key'))) {
                    return back()->withErrors(['error' => 'Paystack API keys not configured in .env file.']);
                }
            } elseif ($paymentMethod->slug === 'flutterwave') {
                if (empty(config('services.flutterwave.secret_key'))) {
                    return back()->withErrors(['error' => 'Flutterwave API keys not configured in .env file.']);
                }
            } elseif ($paymentMethod->slug === 'bank_transfer') {
                $settings = $paymentMethod->settings ?? [];
                if (empty($settings['bank_name']) || empty($settings['account_number'])) {
                    return back()->withErrors(['error' => 'Please configure bank details first.']);
                }
            }
        }
        
        $paymentMethod->update(['is_active' => !$paymentMethod->is_active]);
        
        $status = $paymentMethod->is_active ? 'activated' : 'deactivated';
        
        return back()->with('success', "Payment method {$status} successfully.");
    }
}
