<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\KycFormField;
use Illuminate\Http\Request;

class KycFormBuilderController extends Controller
{
    public function index()
    {
        $fields = KycFormField::orderBy('order')->get();
        return view('admin.kyc-form.index', compact('fields'));
    }

    public function create()
    {
        return view('admin.kyc-form.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'field_name' => 'required|string|max:255|unique:kyc_form_fields,field_name',
            'field_label' => 'required|string|max:255',
            'field_type' => 'required|in:text,textarea,select,file,number,email,tel',
            'field_options' => 'nullable|array',
            'is_required' => 'boolean',
            'is_active' => 'boolean',
            'placeholder' => 'nullable|string|max:255',
            'help_text' => 'nullable|string|max:500',
            'validation_rules' => 'nullable|string|max:255',
        ]);

        // Get the highest order and add 1
        $maxOrder = KycFormField::max('order') ?? 0;
        $validated['order'] = $maxOrder + 1;

        KycFormField::create($validated);

        return redirect()->route('admin.kyc-form.index')->with('success', 'Field added successfully!');
    }

    public function edit(KycFormField $kycForm)
    {
        return view('admin.kyc-form.edit', compact('kycForm'));
    }

    public function update(Request $request, KycFormField $kycForm)
    {
        $validated = $request->validate([
            'field_name' => 'required|string|max:255|unique:kyc_form_fields,field_name,' . $kycForm->id,
            'field_label' => 'required|string|max:255',
            'field_type' => 'required|in:text,textarea,select,file,number,email,tel',
            'field_options' => 'nullable|array',
            'is_required' => 'boolean',
            'is_active' => 'boolean',
            'placeholder' => 'nullable|string|max:255',
            'help_text' => 'nullable|string|max:500',
            'validation_rules' => 'nullable|string|max:255',
        ]);

        $kycForm->update($validated);

        return redirect()->route('admin.kyc-form.index')->with('success', 'Field updated successfully!');
    }

    public function destroy(KycFormField $kycForm)
    {
        $kycForm->delete();
        return back()->with('success', 'Field deleted successfully!');
    }

    public function reorder(Request $request)
    {
        $order = $request->input('order', []);
        
        foreach ($order as $index => $id) {
            KycFormField::where('id', $id)->update(['order' => $index + 1]);
        }

        return response()->json(['success' => true]);
    }
}
