<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\HouseTypeImage;
use App\Models\Property;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class HouseTypeImageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $houseTypeImages = HouseTypeImage::orderBy('house_type')->paginate(20);
        
        // Get all unique house types from properties
        $existingTypes = Property::select('house_type')
            ->distinct()
            ->whereNotNull('house_type')
            ->orderBy('house_type')
            ->pluck('house_type');

        return view('admin.house-types.index', compact('houseTypeImages', 'existingTypes'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get house types that don't have images yet
        $existingImageTypes = HouseTypeImage::pluck('house_type');
        $availableTypes = Property::select('house_type')
            ->distinct()
            ->whereNotNull('house_type')
            ->whereNotIn('house_type', $existingImageTypes)
            ->orderBy('house_type')
            ->pluck('house_type');

        return view('admin.house-types.create', compact('availableTypes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'house_type' => 'required|string|unique:house_type_images,house_type',
            'image' => 'required|image|mimes:jpeg,jpg,png,webp|max:5120',
            'description' => 'nullable|string|max:500',
            'active' => 'boolean',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('house-types', 'public');
        }

        $validated['active'] = $request->has('active');

        HouseTypeImage::create($validated);

        return redirect()->route('admin.house-types.index')
            ->with('success', 'House type image created successfully!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(HouseTypeImage $houseType)
    {
        return view('admin.house-types.edit', compact('houseType'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, HouseTypeImage $houseType)
    {
        $validated = $request->validate([
            'image' => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
            'description' => 'nullable|string|max:500',
            'active' => 'boolean',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($houseType->image) {
                Storage::disk('public')->delete($houseType->image);
            }
            $validated['image'] = $request->file('image')->store('house-types', 'public');
        }

        $validated['active'] = $request->has('active');

        $houseType->update($validated);

        return redirect()->route('admin.house-types.index')
            ->with('success', 'House type image updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(HouseTypeImage $houseType)
    {
        // Delete image
        if ($houseType->image) {
            Storage::disk('public')->delete($houseType->image);
        }

        $houseType->delete();

        return redirect()->route('admin.house-types.index')
            ->with('success', 'House type image deleted successfully!');
    }
}
