<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Car;
use App\Models\CarCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CarController extends Controller
{
    public function index(Request $request)
    {
        $query = Car::with(['vendor', 'category']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('car_category_id', $request->category);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('brand', 'like', "%{$search}%")
                  ->orWhere('model', 'like', "%{$search}%")
                  ->orWhere('plate_number', 'like', "%{$search}%");
            });
        }

        $cars = $query->latest()->paginate(20)->withQueryString();
        $categories = CarCategory::all();

        $stats = [
            'total' => Car::count(),
            'pending' => Car::where('status', 'pending')->count(),
            'approved' => Car::where('status', 'approved')->count(),
            'rejected' => Car::where('status', 'rejected')->count(),
        ];

        return view('admin.cars.index', compact('cars', 'categories', 'stats'));
    }

    public function show(Car $car)
    {
        $car->load(['vendor', 'category', 'bookings', 'reviews']);

        return view('admin.cars.show', compact('car'));
    }

    public function approve(Car $car)
    {
        $car->update([
            'status' => 'approved',
            'rejection_reason' => null,
        ]);

        return back()->with('success', 'Car approved successfully!');
    }

    public function reject(Request $request, Car $car)
    {
        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:500',
        ]);

        $car->update([
            'status' => 'rejected',
            'rejection_reason' => $validated['rejection_reason'],
        ]);

        return back()->with('success', 'Car rejected with reason provided.');
    }

    public function destroy(Car $car)
    {
        // Check if car has bookings
        if ($car->bookings()->count() > 0) {
            return back()->with('error', 'Cannot delete car with existing bookings.');
        }

        // Delete photos
        if ($car->photos) {
            foreach ($car->photos as $photo) {
                Storage::disk('public')->delete($photo);
            }
        }

        $car->delete();

        return redirect()->route('admin.cars.index')
            ->with('success', 'Car deleted successfully!');
    }
}
