<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdPlan;
use Illuminate\Http\Request;

class AdPlanController extends Controller
{
    public function index()
    {
        $adPlans = AdPlan::orderBy('placement_type')->orderBy('price')->get();
        
        return view('admin.ad-plans.index', compact('adPlans'));
    }
    
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'placement_type' => 'required|in:featured,weekend_deals,top_unique',
            'duration_days' => 'required|integer|min:1|max:365',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
        ]);
        
        AdPlan::create($validated);
        
        return back()->with('success', 'Ad plan created successfully.');
    }
    
    public function update(Request $request, AdPlan $adPlan)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'placement_type' => 'required|in:featured,weekend_deals,top_unique',
            'duration_days' => 'required|integer|min:1|max:365',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'active' => 'boolean',
        ]);
        
        $adPlan->update($validated);
        
        return back()->with('success', 'Ad plan updated successfully.');
    }
    
    public function destroy(AdPlan $adPlan)
    {
        // Check if plan is in use
        if ($adPlan->vendorAds()->exists()) {
            return back()->withErrors(['error' => 'Cannot delete ad plan that is in use.']);
        }
        
        $adPlan->delete();
        
        return back()->with('success', 'Ad plan deleted successfully.');
    }
    
    public function toggleStatus(AdPlan $adPlan)
    {
        $adPlan->update(['active' => !$adPlan->active]);
        
        return back()->with('success', 'Ad plan status updated.');
    }
}