<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Property;
use App\Models\Car;
use App\Services\ImageWatermarkService;

class WatermarkExistingImages extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'images:watermark {--type=all : Type of images to watermark (properties, cars, or all)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Add watermark to existing property and car images';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $type = $this->option('type');
        $watermarkService = new ImageWatermarkService();
        
        $this->info('Starting watermark process...');
        $this->newLine();
        
        // Watermark property images
        if ($type === 'all' || $type === 'properties') {
            $this->info('Processing property images...');
            $this->watermarkProperties($watermarkService);
        }
        
        // Watermark car images
        if ($type === 'all' || $type === 'cars') {
            $this->info('Processing car images...');
            $this->watermarkCars($watermarkService);
        }
        
        $this->newLine();
        $this->info('Watermarking completed!');
        
        return Command::SUCCESS;
    }
    
    /**
     * Add watermark to property images
     */
    private function watermarkProperties(ImageWatermarkService $watermarkService)
    {
        $properties = Property::whereNotNull('photos')->get();
        $bar = $this->output->createProgressBar($properties->count());
        $bar->start();
        
        $processed = 0;
        $failed = 0;
        
        foreach ($properties as $property) {
            if (!is_array($property->photos) || empty($property->photos)) {
                $bar->advance();
                continue;
            }
            
            $watermarkedPhotos = [];
            foreach ($property->photos as $photo) {
                try {
                    $watermarkedPath = $watermarkService->addWatermark($photo, 'center', 50);
                    $watermarkedPhotos[] = $watermarkedPath;
                    $processed++;
                } catch (\Exception $e) {
                    $this->error("\nFailed to watermark: {$photo} - " . $e->getMessage());
                    $watermarkedPhotos[] = $photo; // Keep original
                    $failed++;
                }
            }
            
            // Update property with watermarked photos
            $property->update(['photos' => $watermarkedPhotos]);
            
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine();
        $this->info("Properties: {$processed} images watermarked, {$failed} failed");
        $this->newLine();
    }
    
    /**
     * Add watermark to car images
     */
    private function watermarkCars(ImageWatermarkService $watermarkService)
    {
        $cars = Car::whereNotNull('photos')->get();
        $bar = $this->output->createProgressBar($cars->count());
        $bar->start();
        
        $processed = 0;
        $failed = 0;
        
        foreach ($cars as $car) {
            if (!is_array($car->photos) || empty($car->photos)) {
                $bar->advance();
                continue;
            }
            
            $watermarkedPhotos = [];
            foreach ($car->photos as $photo) {
                try {
                    $watermarkedPath = $watermarkService->addWatermark($photo, 'center', 50);
                    $watermarkedPhotos[] = $watermarkedPath;
                    $processed++;
                } catch (\Exception $e) {
                    $this->error("\nFailed to watermark: {$photo} - " . $e->getMessage());
                    $watermarkedPhotos[] = $photo; // Keep original
                    $failed++;
                }
            }
            
            // Update car with watermarked photos
            $car->update(['photos' => $watermarkedPhotos]);
            
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine();
        $this->info("Cars: {$processed} images watermarked, {$failed} failed");
        $this->newLine();
    }
}
