<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CarBooking;
use App\Models\Setting;

class UpdateCarBookingCommissions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'car-bookings:update-commissions {--force : Force recalculation for all bookings}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update commission calculations for existing car bookings and credit vendor balances';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Updating car booking commissions...');
        
        $commissionRate = Setting::get('car_rental_commission_rate', 15);
        $this->info("Using commission rate: {$commissionRate}%");
        
        // Get all car bookings that don't have commission calculated, or all if force flag is set
        if ($this->option('force')) {
            $bookings = CarBooking::all();
            $this->warn("Force flag set - recalculating ALL bookings");
        } else {
            $bookings = CarBooking::where(function($query) {
                $query->whereNull('commission_amount')
                      ->orWhere('commission_amount', 0);
            })->get();
        }
        
        $this->info("Found {$bookings->count()} bookings to update");
        
        $bar = $this->output->createProgressBar($bookings->count());
        $bar->start();
        
        foreach ($bookings as $booking) {
            // Calculate commission (excluding security deposit)
            $booking->commission_rate = $commissionRate;
            $rentalAmount = $booking->total_amount - ($booking->security_deposit ?? 0);
            $booking->commission_amount = ($rentalAmount * $commissionRate) / 100;
            $booking->vendor_payout_amount = $rentalAmount - $booking->commission_amount;
            $booking->save();
            
            // If booking is paid, credit vendor balance
            if ($booking->payment_status === 'paid' && $booking->vendor) {
                $vendor = $booking->vendor;
                
                // Check if vendor balance was already credited with full amount
                // We need to adjust it by removing full amount and adding payout amount
                $this->info("\nAdjusting vendor #{$vendor->id} balance for booking #{$booking->id}");
                $this->info("  Removing: ₦" . number_format($booking->total_amount));
                $this->info("  Adding: ₦" . number_format($booking->vendor_payout_amount));
                
                $vendor->decrement('pending_balance', $booking->total_amount);
                $vendor->decrement('total_earned', $booking->total_amount);
                
                $vendor->increment('pending_balance', $booking->vendor_payout_amount);
                $vendor->increment('total_earned', $booking->vendor_payout_amount);
            }
            
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine();
        $this->info('Commission calculations updated successfully!');
        
        return Command::SUCCESS;
    }
}
